/*
 * Decompiled with CFR 0.152.
 */
package com.gigaspaces.analytics_xtreme.server;

import com.gigaspaces.analytics_xtreme.AnalyticsXtremeConfiguration;
import com.gigaspaces.analytics_xtreme.AnalyticsXtremeStatistics;
import com.gigaspaces.analytics_xtreme.DataLifecyclePolicy;
import com.gigaspaces.analytics_xtreme.internal.TimeAdapter;
import com.gigaspaces.analytics_xtreme.internal.TimeProvider;
import com.gigaspaces.analytics_xtreme.server.AnalyticsXtremeManager;
import com.gigaspaces.analytics_xtreme.spi.BatchDataTarget;
import com.gigaspaces.attribute_store.AttributeStore;
import com.gigaspaces.client.ClearModifiers;
import com.gigaspaces.client.CountModifiers;
import com.gigaspaces.client.iterator.SpaceIterator;
import com.gigaspaces.document.SpaceDocument;
import com.gigaspaces.metadata.SpaceTypeDescriptor;
import com.gigaspaces.metrics.Gauge;
import com.gigaspaces.metrics.LongCounter;
import com.gigaspaces.metrics.Metric;
import com.gigaspaces.metrics.MetricRegistrator;
import com.gigaspaces.query.ISpaceQuery;
import com.gigaspaces.query.QueryResultType;
import com.j_spaces.core.client.SQLQuery;
import java.io.Closeable;
import java.io.IOException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.openspaces.core.GigaSpace;

public class DataLifecycleManager
implements Closeable {
    private static final long ERROR_SLEEP_INTERVAL = Long.getLong("com.gs.analytics_xtreme.manager.error-retry-interval", 5000L);
    private final Logger logger;
    private final GigaSpace gigaSpace;
    private final String tableName;
    private final Clock clock;
    private final String typeName;
    private final String timeColumn;
    private final BatchDataTarget target;
    private final Duration mutabilityPeriod;
    private final Duration batchFeedInterval;
    private final int batchFeedSize;
    private final String confirmedFeedTimestampKey;
    private final Duration evictionThreshold;
    private final Duration evictionPollingInterval;
    private final SQLQuery<SpaceDocument> feedQuery;
    private final SQLQuery<SpaceDocument> evictionQuery;
    private final LongCounter evictedEntries = new LongCounter();
    private final LongCounter copiedEntries = new LongCounter();
    private final LongCounter totalCopiedEntries;
    private final AttributeStore attributeStore;
    private final Object typeInitializedLock = new Object();
    private SpaceTypeDescriptor typeDescriptor;
    private volatile TimeAdapter<?> timeAdapter;
    private volatile Instant confirmedFeedTimestamp;
    private boolean coldStart;

    DataLifecycleManager(DataLifecyclePolicy policy, AnalyticsXtremeManager manager, AnalyticsXtremeConfiguration config) {
        this.gigaSpace = manager.getGigaSpace();
        this.attributeStore = manager.getAttributeStore();
        this.clock = manager.getClock();
        this.totalCopiedEntries = manager.getTotalCopied();
        this.tableName = DataLifecycleManager.getSuffix(policy.getTypeName(), ".");
        this.logger = Logger.getLogger(this.getClass().getName() + "." + this.tableName);
        if (config.isVerbose()) {
            this.logger.setLevel(Level.FINEST);
        }
        this.typeName = policy.getTypeName();
        this.timeColumn = policy.getTimeColumn();
        this.target = policy.getBatchDataTarget();
        Duration speedPeriod = DataLifecycleManager.parseDuration(policy.getSpeedPeriod());
        this.mutabilityPeriod = this.target != null ? DataLifecycleManager.parseDurationOrPercentage(policy.getMutabilityPeriod(), speedPeriod) : null;
        this.batchFeedInterval = this.target != null ? DataLifecycleManager.parseDurationOrPercentage(policy.getBatchFeedInterval(), speedPeriod) : null;
        this.batchFeedSize = policy.getBatchFeedSize();
        this.confirmedFeedTimestampKey = this.target != null ? this.tableName + ".confirmedFeedTimestamp" : null;
        this.evictionThreshold = speedPeriod.plus(DataLifecycleManager.parseDurationOrPercentage(policy.getEvictionBuffer(), speedPeriod));
        this.evictionPollingInterval = DataLifecycleManager.parseDuration(policy.getEvictionPollingInterval());
        this.feedQuery = new SQLQuery(this.typeName, this.timeColumn + " > ? AND " + this.timeColumn + " <= ?", QueryResultType.DOCUMENT);
        this.evictionQuery = new SQLQuery(this.typeName, this.timeColumn + " < ?");
        if (this.logger.isLoggable(Level.INFO)) {
            this.logger.info(String.format("DataLifecycleManager created: timeColumn=%s, speedPeriod=%s, mutabilityPeriod=%s, batchFeedInterval=%s, batchFeedSize=%s, evictionThreshold=%s, evictionPollingInterval=%s", this.timeColumn, TimeProvider.format(speedPeriod), TimeProvider.format(this.mutabilityPeriod), TimeProvider.format(this.batchFeedInterval), this.batchFeedSize, TimeProvider.format(this.evictionThreshold), TimeProvider.format(this.evictionPollingInterval)));
        }
    }

    private static Duration parseDuration(String s) {
        try {
            return Duration.parse(s);
        }
        catch (DateTimeParseException e) {
            throw new DateTimeParseException(e.getMessage() + ": " + e.getParsedString(), e.getParsedString(), e.getErrorIndex());
        }
    }

    public Duration getEvictionThreshold() {
        return this.evictionThreshold;
    }

    public Duration getEvictionPollingInterval() {
        return this.evictionPollingInterval;
    }

    public String getTableName() {
        return this.tableName;
    }

    public Duration getMutabilityPeriod() {
        return this.mutabilityPeriod;
    }

    public Duration getBatchFeedInterval() {
        return this.batchFeedInterval;
    }

    public long getEvictedEntries() {
        return this.evictedEntries.getCount();
    }

    public AnalyticsXtremeStatistics.TypeStatistics getStatistics() {
        return new AnalyticsXtremeStatistics.TypeStatistics.Builder().statistics("confirmedFeedTimestamp", this.confirmedFeedTimestamp).statistics("copied", Long.valueOf(this.copiedEntries.getCount())).statistics("evicted", Long.valueOf(this.evictedEntries.getCount())).build();
    }

    boolean hasBatchTarget() {
        return this.target != null;
    }

    void setColdStart(boolean coldStart) {
        this.coldStart = coldStart;
    }

    void feed() {
        this.logger.info("Batch processing activated");
        try {
            this.confirmedFeedTimestamp = this.initConfirmedFeedTimestamp();
        }
        catch (InterruptedException e) {
            this.logger.warning("Batch processing was interrupted");
            Thread.currentThread().interrupt();
            return;
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Unexpected exception while initializing confirmedFeedTimestamp", e);
            return;
        }
        while (true) {
            try {
                while (true) {
                    Instant threshold;
                    Instant nextFeedTimestamp;
                    if ((nextFeedTimestamp = this.confirmedFeedTimestamp.plus(this.batchFeedInterval)).isAfter(threshold = this.clock.instant().minus(this.mutabilityPeriod))) {
                        Duration sleepDuration = Duration.between(threshold, nextFeedTimestamp);
                        if (this.logger.isLoggable(Level.FINE)) {
                            this.logger.fine(String.format("Next feed: %s to %s - exceeds mutability threshold (%s) - sleeping for %s seconds", this.confirmedFeedTimestamp, nextFeedTimestamp, threshold, Float.valueOf((float)sleepDuration.toMillis() / 1000.0f)));
                        }
                        Thread.sleep(sleepDuration.toMillis());
                    }
                    this.feed(this.confirmedFeedTimestamp, nextFeedTimestamp);
                    this.confirmedFeedTimestamp = this.updateConfirmedFeedTimestamp(nextFeedTimestamp);
                }
            }
            catch (Exception e) {
                if (this.isInterrupted(e)) {
                    this.logInterrupted(e);
                    Thread.currentThread().interrupt();
                    return;
                }
                this.logger.log(Level.SEVERE, "Exception during batch processing: " + e.getMessage(), e);
                this.logger.info("Sleeping " + ERROR_SLEEP_INTERVAL + "ms before retry");
                try {
                    Thread.sleep(ERROR_SLEEP_INTERVAL);
                }
                catch (InterruptedException e1) {
                    this.logInterrupted(e1);
                    Thread.currentThread().interrupt();
                    return;
                }
            }
        }
    }

    private void logInterrupted(Throwable e) {
        if (this.logger.isLoggable(Level.FINE)) {
            this.logger.log(Level.FINE, "Batch processing was interrupted", e);
        } else {
            this.logger.warning("Batch processing was interrupted");
        }
    }

    private boolean isInterrupted(Throwable e) {
        if (e instanceof InterruptedException) {
            return true;
        }
        Throwable cause = e.getCause();
        return cause != null && this.isInterrupted(cause);
    }

    void registerMetrics(MetricRegistrator metricRegistrator) {
        metricRegistrator.register("evicted-entries", (Metric)this.evictedEntries);
        metricRegistrator.register("copied-entries", (Metric)this.copiedEntries);
        metricRegistrator.register("confirmed-feed-timestamp", (Metric)new Gauge<String>(){

            public String getValue() throws Exception {
                return String.valueOf(DataLifecycleManager.this.confirmedFeedTimestamp);
            }
        });
    }

    private Instant updateConfirmedFeedTimestamp(Instant timestamp) throws InterruptedException {
        while (true) {
            try {
                this.attributeStore.set(this.confirmedFeedTimestampKey, timestamp.toString());
                return timestamp;
            }
            catch (IOException e) {
                this.logger.warning("Failed to update " + this.confirmedFeedTimestampKey + " to " + timestamp + ": " + e.getMessage());
                Thread.sleep(1000L);
                continue;
            }
            break;
        }
    }

    private Instant initConfirmedFeedTimestamp() throws InterruptedException {
        SpaceDocument entry;
        long retryInterval = 1000L;
        while (this.getTypeDescriptor() == null) {
            Thread.sleep(1000L);
        }
        Instant result = this.loadConfirmedFeedTimestamp(1000L);
        if (result != null) {
            this.logger.info("Restored confirmedFeedTimestamp from attribute store: " + result);
            return result;
        }
        this.logger.info("ConfirmedFeedTimestamp is empty - searching for oldest entry...");
        SQLQuery query = new SQLQuery(this.typeName, this.timeColumn + " > ?", QueryResultType.DOCUMENT, new Object[]{this.timeAdapter.fromInstant(Instant.EPOCH)});
        while ((entry = (SpaceDocument)this.gigaSpace.read((ISpaceQuery)query)) == null) {
            Thread.sleep(1000L);
        }
        Instant oldest = this.timeAdapter.toInstant((Comparable)entry.getProperty(this.timeColumn));
        result = oldest.minusSeconds(1L);
        this.logger.info("Timestamp of oldest entry is " + oldest + ", setting confirmedFeedTimestamp to " + result);
        return result;
    }

    private Instant loadConfirmedFeedTimestamp(long retryInterval) throws InterruptedException {
        if (this.coldStart) {
            this.coldStart = false;
            this.logger.info("Activated with coldStart - ignoring previous confirmedFeedTimestamp");
            return null;
        }
        while (true) {
            try {
                String s = this.attributeStore.get(this.confirmedFeedTimestampKey);
                return s != null ? Instant.parse(s) : null;
            }
            catch (IOException e) {
                this.logger.warning("Failed to retrieve " + this.confirmedFeedTimestampKey + " from attribute store: " + e.getMessage());
                Thread.sleep(retryInterval);
                continue;
            }
            break;
        }
    }

    private void feed(Instant from, Instant to) {
        if (this.logger.isLoggable(Level.FINE)) {
            this.logger.fine("Feeding entries from " + from + " to " + to);
        }
        this.feedQuery.setParameter(1, this.timeAdapter.fromInstant(from)).setParameter(2, this.timeAdapter.fromInstant(to));
        int numOfEntries = 0;
        ArrayList<SpaceDocument> batch = new ArrayList<SpaceDocument>(this.batchFeedSize);
        try (SpaceIterator iterator = this.gigaSpace.iterator(this.feedQuery, this.batchFeedSize);){
            for (SpaceDocument entry : iterator) {
                batch.add(entry);
                ++numOfEntries;
                if (batch.size() != this.batchFeedSize) continue;
                this.target.feed(batch, this.tableName, this.getTypeDescriptor());
                batch.clear();
            }
        }
        if (!batch.isEmpty()) {
            this.target.feed(batch, this.tableName, this.getTypeDescriptor());
        }
        this.copiedEntries.inc((long)numOfEntries);
        this.totalCopiedEntries.inc((long)numOfEntries);
        if (this.logger.isLoggable(Level.INFO)) {
            this.logger.info(String.format("Copied %s entries between %s and %s to batch", numOfEntries, from, to));
        }
    }

    int evict(Instant currTime) {
        Instant threshold = currTime.minus(this.evictionThreshold);
        String evictionBlocker = this.checkEvictionValid(threshold);
        if (evictionBlocker != null) {
            if (this.logger.isLoggable(Level.INFO)) {
                this.logger.info("Eviction skipped - " + evictionBlocker);
            }
            return 0;
        }
        try {
            int remaining;
            this.evictionQuery.setParameter(1, this.timeAdapter.fromInstant(threshold));
            int evicted = this.gigaSpace.clear(this.evictionQuery, ClearModifiers.NONE);
            this.evictedEntries.inc((long)evicted);
            if (this.logger.isLoggable(Level.FINE)) {
                this.logger.fine(String.format("Evicted %s entries below %s", evicted, threshold));
            }
            if ((remaining = this.gigaSpace.count(this.evictionQuery, CountModifiers.DIRTY_READ)) != 0) {
                this.logger.warning(remaining + " entries survived eviction");
            }
            return evicted;
        }
        catch (Exception e) {
            this.logger.warning("Eviction failed: " + e.getMessage());
            return 0;
        }
    }

    private String checkEvictionValid(Instant threshold) {
        if (this.getTypeDescriptor() == null) {
            return "type is not registered in space";
        }
        if (this.hasBatchTarget()) {
            Instant timestamp = this.confirmedFeedTimestamp;
            if (timestamp == null) {
                return "confirmed feed timestamp has not been initialized yet";
            }
            if (threshold.isAfter(timestamp)) {
                return String.format("cannot evict %s because confirmed feed timestamp is %s", threshold, timestamp);
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private SpaceTypeDescriptor getTypeDescriptor() {
        if (this.typeDescriptor == null) {
            Object object = this.typeInitializedLock;
            synchronized (object) {
                if (this.typeDescriptor == null) {
                    this.typeDescriptor = this.gigaSpace.getTypeManager().getTypeDescriptor(this.typeName);
                    if (this.typeDescriptor != null) {
                        this.timeAdapter = TimeAdapter.of(this.typeDescriptor.getFixedProperty(this.timeColumn).getType());
                    }
                }
            }
        }
        return this.typeDescriptor;
    }

    private static Duration parseDurationOrPercentage(String s, Duration baseDuration) {
        Integer percentage = s.endsWith("%") ? Integer.valueOf(Integer.parseInt(s.substring(0, s.length() - 1))) : null;
        return percentage != null ? baseDuration.multipliedBy(percentage.intValue()).dividedBy(100L) : DataLifecycleManager.parseDuration(s);
    }

    private static String getSuffix(String s, String token) {
        int pos = s.lastIndexOf(token);
        return pos == -1 ? s : s.substring(pos + token.length());
    }

    @Override
    public void close() throws IOException {
        if (this.target != null) {
            this.target.close();
        }
    }
}

