/*
 * Decompiled with CFR 0.152.
 */
package org.openspaces.grid.gsm.rebalancing;

import com.gigaspaces.cluster.activeelection.SpaceMode;
import com.sun.jini.admin.DestroyAdmin;
import java.io.File;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.openspaces.admin.Admin;
import org.openspaces.admin.AdminException;
import org.openspaces.admin.GridComponent;
import org.openspaces.admin.gsa.GridServiceAgent;
import org.openspaces.admin.gsa.GridServiceAgents;
import org.openspaces.admin.gsc.GridServiceContainer;
import org.openspaces.admin.internal.admin.InternalAdmin;
import org.openspaces.admin.internal.pu.InternalProcessingUnit;
import org.openspaces.admin.machine.Machine;
import org.openspaces.admin.pu.DeploymentStatus;
import org.openspaces.admin.pu.ProcessingUnit;
import org.openspaces.admin.pu.ProcessingUnitInstance;
import org.openspaces.admin.space.SpaceInstance;
import org.openspaces.core.GigaSpace;
import org.openspaces.core.internal.commons.math.fraction.Fraction;
import org.openspaces.grid.gsm.capacity.CapacityRequirements;
import org.openspaces.grid.gsm.capacity.CapacityRequirementsPerAgent;
import org.openspaces.grid.gsm.capacity.CpuCapacityRequirement;
import org.openspaces.grid.gsm.containers.ContainersSlaUtils;
import org.openspaces.grid.gsm.rebalancing.FutureStatefulProcessingUnitInstance;
import org.openspaces.grid.gsm.rebalancing.FutureStatelessProcessingUnitInstance;
import org.openspaces.grid.gsm.rebalancing.exceptions.RemovedContainerProcessingUnitDeploymentException;
import org.openspaces.grid.gsm.rebalancing.exceptions.WrongContainerProcessingUnitRelocationException;

public class RebalancingUtils {
    private static final Log logger = LogFactory.getLog(RebalancingUtils.class);

    static Collection<FutureStatelessProcessingUnitInstance> incrementNumberOfStatelessInstancesAsync(final ProcessingUnit pu, GridServiceContainer[] containers, final Log logger, long duration, TimeUnit timeUnit) {
        if (pu.getMaxInstancesPerVM() != 1) {
            throw new IllegalArgumentException("Only one instance per VM is allowed");
        }
        List<GridServiceContainer> unusedContainers = RebalancingUtils.getUnusedContainers(pu, containers);
        final Admin admin = pu.getAdmin();
        HashMap<GridServiceContainer, 1> futureInstances = new HashMap<GridServiceContainer, 1>();
        final AtomicInteger targetNumberOfInstances = new AtomicInteger(pu.getNumberOfInstances());
        final long start = System.currentTimeMillis();
        final long end = start + timeUnit.toMillis(duration);
        Iterator<GridServiceContainer> iterator = unusedContainers.iterator();
        while (iterator.hasNext()) {
            GridServiceContainer container;
            final GridServiceContainer targetContainer = container = iterator.next();
            futureInstances.put(container, new FutureStatelessProcessingUnitInstance(){
                AtomicReference<Throwable> throwable = new AtomicReference();
                ProcessingUnitInstance newInstance;

                public boolean isTimedOut() {
                    return System.currentTimeMillis() > end;
                }

                public boolean isDone() {
                    this.end();
                    return this.isTimedOut() || this.throwable.get() != null || this.newInstance != null;
                }

                public ProcessingUnitInstance get() throws ExecutionException, IllegalStateException, TimeoutException {
                    this.end();
                    if (this.getException() != null) {
                        throw this.getException();
                    }
                    if (this.newInstance == null) {
                        if (this.isTimedOut()) {
                            throw new TimeoutException("Relocation timeout");
                        }
                        throw new IllegalStateException("Async operation is not done yet.");
                    }
                    return this.newInstance;
                }

                public Date getTimestamp() {
                    return new Date(start);
                }

                public ExecutionException getException() {
                    this.end();
                    Throwable t = this.throwable.get();
                    if (t != null) {
                        return new ExecutionException(t.getMessage(), t);
                    }
                    return null;
                }

                @Override
                public GridServiceContainer getTargetContainer() {
                    return targetContainer;
                }

                @Override
                public ProcessingUnit getProcessingUnit() {
                    return pu;
                }

                @Override
                public String getFailureMessage() throws IllegalStateException {
                    if (this.isTimedOut()) {
                        return "deployment timeout of processing unit " + pu.getName() + " on " + RebalancingUtils.gscToString(targetContainer);
                    }
                    if (this.getException() != null) {
                        return this.getException().getMessage();
                    }
                    throw new IllegalStateException("Relocation has not encountered any failure.");
                }

                private void end() {
                    if (!targetContainer.isDiscovered()) {
                        this.throwable.set(new RemovedContainerProcessingUnitDeploymentException(pu, targetContainer));
                    } else if (this.throwable.get() == null && this.newInstance == null) {
                        this.incrementInstance();
                        ProcessingUnitInstance[] instances = targetContainer.getProcessingUnitInstances(pu.getName());
                        if (instances.length > 0) {
                            this.newInstance = instances[0];
                        }
                    }
                }

                private void incrementInstance() {
                    int maxNumberOfInstances;
                    final String uuid = "[incrementUid:" + UUID.randomUUID().toString() + "] ";
                    int numberOfInstances = pu.getNumberOfInstances();
                    if (numberOfInstances < (maxNumberOfInstances = RebalancingUtils.getContainersOnMachines(pu).length)) {
                        if (targetNumberOfInstances.get() == numberOfInstances + 1) {
                            if (logger.isInfoEnabled()) {
                                logger.info((Object)("Waiting for pu.numberOfInstances to increment from " + numberOfInstances + " to " + targetNumberOfInstances.get() + ". Number of relevant containers " + maxNumberOfInstances));
                            }
                        } else if (admin.getGridServiceManagers().getSize() > 1 && !((InternalProcessingUnit)pu).isBackupGsmInSync()) {
                            if (logger.isInfoEnabled()) {
                                logger.info((Object)"Waiting for backup gsm to sync with active gsm");
                            }
                        } else {
                            targetNumberOfInstances.set(numberOfInstances + 1);
                            if (logger.isInfoEnabled()) {
                                logger.info((Object)(uuid + " Planning to increment pu.numberOfInstances from " + numberOfInstances + " to " + targetNumberOfInstances.get() + ". Number of relevant containers " + maxNumberOfInstances));
                            }
                            ((InternalAdmin)admin).scheduleAdminOperation(new Runnable(){

                                @Override
                                public void run() {
                                    try {
                                        pu.incrementInstance();
                                        if (logger.isInfoEnabled()) {
                                            logger.info((Object)(uuid + " pu.incrementInstance() called"));
                                        }
                                    }
                                    catch (AdminException e) {
                                        throwable.set(e);
                                    }
                                    catch (Throwable e) {
                                        logger.error((Object)(uuid + " Unexpected Exception: " + e.getMessage()), e);
                                        throwable.set(e);
                                    }
                                }
                            });
                        }
                    }
                }
            });
        }
        return futureInstances.values();
    }

    private static List<GridServiceContainer> getUnusedContainers(ProcessingUnit pu, GridServiceContainer[] containers) {
        ArrayList<GridServiceContainer> unusedContainers = new ArrayList<GridServiceContainer>();
        for (GridServiceContainer container : containers) {
            if (container.getProcessingUnitInstances(pu.getName()).length != 0) continue;
            unusedContainers.add(container);
        }
        return unusedContainers;
    }

    public static String puInstancesToString(Collection<ProcessingUnitInstance> instances) {
        StringBuilder builder = new StringBuilder();
        for (ProcessingUnitInstance instance : instances) {
            builder.append(RebalancingUtils.puInstanceToString(instance));
            builder.append(File.separator);
        }
        return builder.toString();
    }

    static FutureStatefulProcessingUnitInstance relocateProcessingUnitInstanceAsync(final GridServiceContainer targetContainer, final ProcessingUnitInstance puInstance, final Log logger, long duration, TimeUnit timeUnit) {
        final ProcessingUnit pu = puInstance.getProcessingUnit();
        final GridServiceContainer[] replicationSourceContainers = RebalancingUtils.getReplicationSourceContainers(puInstance);
        final int instanceId = puInstance.getInstanceId();
        final AtomicReference relocateThrowable = new AtomicReference();
        final Admin admin = puInstance.getAdmin();
        final int runningNumber = puInstance.getClusterInfo().getRunningNumber();
        final String puName = puInstance.getName();
        final GridServiceContainer sourceContainer = puInstance.getGridServiceContainer();
        final Set<ProcessingUnitInstance> puInstancesFromSamePartition = RebalancingUtils.getOtherInstancesFromSamePartition(puInstance);
        if (logger.isDebugEnabled()) {
            logger.debug((Object)("Found instances from the same partition as " + RebalancingUtils.puInstanceToString(puInstance) + " : " + RebalancingUtils.puInstancesToString(puInstancesFromSamePartition)));
        }
        if (puInstancesFromSamePartition.size() != pu.getNumberOfBackups()) {
            RebalancingUtils.destroyExtraInstanceOfSamePartition(logger, pu, puInstancesFromSamePartition);
            throw new IllegalStateException("puInstancesFromSamePartition has " + puInstancesFromSamePartition.size() + " instances instead of " + pu.getNumberOfBackups());
        }
        final long start = System.currentTimeMillis();
        final long end = start + timeUnit.toMillis(duration);
        ((InternalAdmin)admin).scheduleAdminOperation(new Runnable(){

            @Override
            public void run() {
                try {
                    logger.debug((Object)("Relocation of " + RebalancingUtils.puInstanceToString(puInstance) + " to " + ContainersSlaUtils.gscToString(targetContainer) + " has started."));
                    puInstance.relocate(targetContainer);
                }
                catch (AdminException e) {
                    logger.error((Object)("Admin exception " + e.getMessage()), (Throwable)e);
                    relocateThrowable.set(e);
                }
                catch (Throwable e) {
                    logger.error((Object)("Unexpected exception " + e.getMessage()), e);
                    relocateThrowable.set(e);
                }
            }
        });
        return new FutureStatefulProcessingUnitInstance(){
            Throwable throwable;
            ProcessingUnitInstance newInstance;

            public boolean isTimedOut() {
                return System.currentTimeMillis() > end;
            }

            public boolean isDone() {
                this.endRelocation();
                return this.isTimedOut() || this.throwable != null || this.newInstance != null;
            }

            public ProcessingUnitInstance get() throws ExecutionException, IllegalStateException, TimeoutException {
                this.endRelocation();
                ExecutionException exception = this.getException();
                if (exception != null) {
                    throw exception;
                }
                if (this.newInstance == null) {
                    if (this.isTimedOut()) {
                        throw new TimeoutException("Relocation timeout");
                    }
                    throw new IllegalStateException("Async operation is not done yet.");
                }
                return this.newInstance;
            }

            public Date getTimestamp() {
                return new Date(start);
            }

            public ExecutionException getException() {
                this.endRelocation();
                if (this.throwable != null) {
                    return new ExecutionException(this.throwable.getMessage(), this.throwable);
                }
                return null;
            }

            private void endRelocation() {
                boolean inProgress = true;
                this.tryStateChange();
                if (this.newInstance != null || this.throwable != null) {
                    inProgress = false;
                }
                if (inProgress && logger.isDebugEnabled()) {
                    logger.debug((Object)("Relocation from " + ContainersSlaUtils.gscToString(this.getSourceContainer()) + " to " + ContainersSlaUtils.gscToString(this.getTargetContainer()) + " is in progress."));
                }
            }

            private void tryStateChange() {
                ProcessingUnitInstance relocatedInstance = this.getRelocatedProcessingUnitInstance();
                if (relocatedInstance != null) {
                    if (relocatedInstance.getGridServiceContainer().equals(targetContainer)) {
                        if (relocatedInstance.getSpaceInstance() != null && relocatedInstance.getSpaceInstance().getMode() != SpaceMode.NONE) {
                            if (logger.isDebugEnabled()) {
                                logger.debug((Object)("Relocation from " + ContainersSlaUtils.gscToString(this.getSourceContainer()) + " to " + ContainersSlaUtils.gscToString(this.getTargetContainer()) + " had ended successfully."));
                            }
                            this.newInstance = relocatedInstance;
                        }
                    } else {
                        if (logger.isDebugEnabled()) {
                            logger.debug((Object)("Relocation from " + ContainersSlaUtils.gscToString(this.getSourceContainer()) + " to " + ContainersSlaUtils.gscToString(this.getTargetContainer()) + " has ended with an error."));
                        }
                        this.throwable = new WrongContainerProcessingUnitRelocationException(puInstance, targetContainer);
                    }
                }
            }

            private ProcessingUnitInstance getRelocatedProcessingUnitInstance() {
                for (GridServiceContainer container : admin.getGridServiceContainers()) {
                    for (ProcessingUnitInstance instance : container.getProcessingUnitInstances(puName)) {
                        if (instance.equals(puInstance) || instance.getClusterInfo().getRunningNumber() != runningNumber || puInstancesFromSamePartition.contains(instance)) continue;
                        return instance;
                    }
                }
                return null;
            }

            private boolean isAtLeastOneInstanceValid(Set<ProcessingUnitInstance> instances) {
                boolean isValidState = false;
                for (ProcessingUnitInstance instance : instances) {
                    if (!instance.isDiscovered() || !instance.getGridServiceContainer().isDiscovered()) continue;
                    isValidState = true;
                    break;
                }
                return isValidState;
            }

            @Override
            public String getFailureMessage() {
                if (this.isTimedOut()) {
                    return "relocation timeout of processing unit instance " + instanceId + " from " + RebalancingUtils.gscToString(sourceContainer) + " to " + RebalancingUtils.gscToString(targetContainer);
                }
                if (this.getException() != null) {
                    return this.getException().getMessage();
                }
                throw new IllegalStateException("Relocation has not encountered any failure.");
            }

            @Override
            public GridServiceContainer getTargetContainer() {
                return targetContainer;
            }

            @Override
            public ProcessingUnit getProcessingUnit() {
                return pu;
            }

            @Override
            public int getInstanceId() {
                return instanceId;
            }

            @Override
            public GridServiceContainer getSourceContainer() {
                return sourceContainer;
            }

            @Override
            public GridServiceContainer[] getReplicaitonSourceContainers() {
                return replicationSourceContainers;
            }
        };
    }

    private static void destroyExtraInstanceOfSamePartition(Log logger, ProcessingUnit pu, Set<ProcessingUnitInstance> puInstancesFromSamePartition) {
        for (ProcessingUnitInstance instance : puInstancesFromSamePartition) {
            SpaceInstance spaceInstance = instance.getSpaceInstance();
            if (spaceInstance == null || spaceInstance.getMode() != SpaceMode.NONE) continue;
            logger.info((Object)(RebalancingUtils.puInstanceToString(instance) + " has " + puInstancesFromSamePartition.size() + " instances instead of " + pu.getNumberOfBackups() + " - destroying stale instance on " + RebalancingUtils.gscToString(instance.getGridServiceContainer())));
            try {
                instance.destroy();
            }
            catch (AdminException ex) {
                GigaSpace gigaSpace = spaceInstance.getGigaSpace();
                if (gigaSpace != null && gigaSpace.getSpace() != null) {
                    try {
                        ((DestroyAdmin)gigaSpace.getSpace().getAdmin()).destroy();
                    }
                    catch (RemoteException re) {
                        logger.warn((Object)("Failed to destroy instance " + RebalancingUtils.puInstanceToString(instance)), (Throwable)re);
                    }
                    continue;
                }
                logger.warn((Object)("Failed to destroy instance " + RebalancingUtils.puInstanceToString(instance)), (Throwable)ex);
            }
        }
    }

    public static GridServiceContainer[] getReplicationSourceContainers(ProcessingUnitInstance instance) {
        HashSet<GridServiceContainer> repContainers = new HashSet<GridServiceContainer>();
        GridServiceContainer[] containers = instance.getAdmin().getGridServiceContainers().getContainers();
        int numberOfBackups = instance.getProcessingUnit().getNumberOfBackups();
        if (numberOfBackups == 0) {
            return new GridServiceContainer[0];
        }
        if (!RebalancingUtils.isProcessingUnitPartitionIntact(instance.getProcessingUnit(), instance.getInstanceId(), containers)) {
            throw new IllegalStateException("Cannot relocate pu instance " + RebalancingUtils.puInstanceToString(instance) + " since partition is not intact.");
        }
        for (int backupId = 0; backupId <= numberOfBackups; ++backupId) {
            if (backupId == instance.getBackupId()) continue;
            repContainers.add(RebalancingUtils.findProcessingUnitInstance(instance.getProcessingUnit(), instance.getInstanceId(), backupId, containers).getGridServiceContainer());
        }
        return repContainers.toArray(new GridServiceContainer[repContainers.size()]);
    }

    public static boolean isProcessingUnitIntact(ProcessingUnit pu, GridServiceContainer[] containers) {
        boolean intact = true;
        if (pu.getStatus() != DeploymentStatus.INTACT) {
            intact = false;
        } else {
            int allContainers = pu.getAdmin().getGridServiceContainers().getContainers().length;
            if (pu.getNumberOfBackups() > 0) {
                for (int instanceId = 1; intact && instanceId <= pu.getNumberOfInstances(); ++instanceId) {
                    if (RebalancingUtils.isProcessingUnitPartitionIntact(pu, instanceId, containers)) continue;
                    if (logger.isTraceEnabled()) {
                        logger.trace((Object)(pu.getName() + " missing instance with id [" + instanceId + "] containers [" + containers.length + "/" + allContainers + "]"));
                    }
                    intact = false;
                    break;
                }
            } else {
                ProcessingUnitInstance[] instances = pu.getInstances();
                if (instances.length < pu.getNumberOfInstances()) {
                    if (logger.isTraceEnabled()) {
                        logger.trace((Object)(pu.getName() + " missing instances, expected: " + pu.getNumberOfInstances() + " actual: " + instances.length + " containers [" + containers.length + "/" + allContainers + "]"));
                    }
                    intact = false;
                } else {
                    for (ProcessingUnitInstance instance : instances) {
                        if (RebalancingUtils.findProcessingUnitInstance(pu, instance.getInstanceId(), 0, containers) != null) continue;
                        if (logger.isTraceEnabled()) {
                            logger.trace((Object)(pu.getName() + " missing instance [" + instance.getProcessingUnitInstanceName() + "] containers [" + containers.length + "/" + allContainers + "]"));
                        }
                        intact = false;
                        break;
                    }
                }
            }
        }
        return intact;
    }

    public static String getProcessingUnitIntactDescription(ProcessingUnit pu, GridServiceContainer[] containers) {
        boolean intact = true;
        if (pu.getStatus() != DeploymentStatus.INTACT) {
            return pu.getName() + " Status = " + (Object)((Object)pu.getStatus());
        }
        int allContainers = pu.getAdmin().getGridServiceContainers().getContainers().length;
        if (pu.getNumberOfBackups() > 0) {
            for (int instanceId = 1; intact && instanceId <= pu.getNumberOfInstances(); ++instanceId) {
                if (RebalancingUtils.isProcessingUnitPartitionIntact(pu, instanceId, containers)) continue;
                return pu.getName() + " missing instance with id [" + instanceId + "] containers [" + containers.length + "/" + allContainers + "]";
            }
        } else {
            ProcessingUnitInstance[] instances = pu.getInstances();
            if (instances.length < pu.getNumberOfInstances()) {
                return pu.getName() + " missing instances, expected: " + pu.getNumberOfInstances() + " actual: " + instances.length + " containers [" + containers.length + "/" + allContainers + "]";
            }
            for (ProcessingUnitInstance instance : instances) {
                if (RebalancingUtils.findProcessingUnitInstance(pu, instance.getInstanceId(), 0, containers) != null) continue;
                return pu.getName() + " missing instance [" + instance.getProcessingUnitInstanceName() + "] containers [" + containers.length + "/" + allContainers + "]";
            }
        }
        return pu.getName() + " was found to be intact, but Status = " + (Object)((Object)pu.getStatus());
    }

    public static boolean isProcessingUnitHasMinimumNumberOfInstancesPerPartition(ProcessingUnit pu, int minimumNumberOfInstancesPerPartition) {
        return RebalancingUtils.isProcessingUnitHasMinimumNumberOfInstancesPerPartition(pu, pu.getAdmin().getGridServiceContainers().getContainers(), minimumNumberOfInstancesPerPartition);
    }

    private static boolean isProcessingUnitHasMinimumNumberOfInstancesPerPartition(ProcessingUnit pu, GridServiceContainer[] containers, int minimumNumberOfInstancesPerPartition) {
        boolean hasMinimum = true;
        if (pu.getNumberOfBackups() > 0) {
            for (int instanceId = 1; hasMinimum && instanceId <= pu.getNumberOfInstances(); ++instanceId) {
                if (RebalancingUtils.isProcessingUnitPartitionHasMinimumNumberOfInstances(pu, instanceId, containers, minimumNumberOfInstancesPerPartition)) continue;
                hasMinimum = false;
                break;
            }
        } else {
            hasMinimum = RebalancingUtils.isProcessingUnitIntact(pu, containers);
        }
        return hasMinimum;
    }

    public static boolean isProcessingUnitPartitionIntact(ProcessingUnitInstance instance) {
        GridServiceContainer[] containers = instance.getAdmin().getGridServiceContainers().getContainers();
        return RebalancingUtils.isProcessingUnitPartitionIntact(instance.getProcessingUnit(), instance.getInstanceId(), containers);
    }

    public static boolean isProcessingUnitIntact(ProcessingUnit pu) {
        return RebalancingUtils.isProcessingUnitIntact(pu, pu.getAdmin().getGridServiceContainers().getContainers());
    }

    private static ProcessingUnitInstance findProcessingUnitInstance(ProcessingUnit pu, int instanceId, int backupId, GridServiceContainer[] containers) {
        for (GridServiceContainer container : containers) {
            for (ProcessingUnitInstance instance : container.getProcessingUnitInstances(pu.getName())) {
                if (instance.getInstanceId() != instanceId || instance.getBackupId() != backupId) continue;
                return instance;
            }
        }
        return null;
    }

    public static boolean isProcessingUnitPartitionIntact(ProcessingUnit pu, int instanceId, GridServiceContainer[] containers) {
        return RebalancingUtils.isProcessingUnitPartitionHasMinimumNumberOfInstances(pu, instanceId, containers, 1 + pu.getNumberOfBackups());
    }

    private static boolean isProcessingUnitPartitionHasMinimumNumberOfInstances(ProcessingUnit pu, int instanceId, GridServiceContainer[] containers, int minimumNumberOfInstancesPerPartition) {
        boolean hasMinimum = true;
        if (minimumNumberOfInstancesPerPartition >= 1) {
            int numberOfPrimaryInstances = 0;
            int numberOfBackupInstances = 0;
            for (int backupId = 0; backupId <= pu.getNumberOfBackups(); ++backupId) {
                ProcessingUnitInstance instance = RebalancingUtils.findProcessingUnitInstance(pu, instanceId, backupId, containers);
                if (instance == null || instance.getSpaceInstance() == null) continue;
                if (instance.getSpaceInstance().getMode() == SpaceMode.BACKUP) {
                    ++numberOfBackupInstances;
                    continue;
                }
                if (instance.getSpaceInstance().getMode() != SpaceMode.PRIMARY) continue;
                ++numberOfPrimaryInstances;
            }
            hasMinimum = numberOfPrimaryInstances == 1 && 1 + numberOfBackupInstances >= minimumNumberOfInstancesPerPartition;
        }
        return hasMinimum;
    }

    public static Set<ProcessingUnitInstance> getOtherInstancesFromSamePartition(ProcessingUnitInstance instance) {
        HashSet<ProcessingUnitInstance> puInstancesFromSamePartition = new HashSet<ProcessingUnitInstance>();
        for (GridServiceContainer container : instance.getAdmin().getGridServiceContainers()) {
            puInstancesFromSamePartition.addAll(RebalancingUtils.getOtherInstancesFromSamePartitionInContainer(container, instance));
        }
        return puInstancesFromSamePartition;
    }

    public static Set<ProcessingUnitInstance> getOtherInstancesFromSamePartitionInContainer(GridServiceContainer container, ProcessingUnitInstance instance) {
        HashSet<ProcessingUnitInstance> puInstancesFromSamePartition = new HashSet<ProcessingUnitInstance>();
        for (ProcessingUnitInstance instanceOnContainer : container.getProcessingUnitInstances(instance.getName())) {
            if (instanceOnContainer.getInstanceId() != instance.getInstanceId() || instanceOnContainer.equals(instance)) continue;
            puInstancesFromSamePartition.add(instanceOnContainer);
        }
        return puInstancesFromSamePartition;
    }

    public static Set<ProcessingUnitInstance> getOtherInstancesFromSamePartitionInMachine(Machine machine, ProcessingUnitInstance puInstance) {
        HashSet<ProcessingUnitInstance> puInstancesFromSamePartition = new HashSet<ProcessingUnitInstance>();
        for (GridServiceContainer container : machine.getGridServiceContainers()) {
            puInstancesFromSamePartition.addAll(RebalancingUtils.getOtherInstancesFromSamePartitionInContainer(container, puInstance));
        }
        return puInstancesFromSamePartition;
    }

    public static boolean isEvenlyDistributedAcrossMachines(ProcessingUnit pu, CapacityRequirementsPerAgent aggregatedAllocatedCapacity) {
        boolean isEvenlyDistributedAcrossMachines = true;
        Machine[] machines = RebalancingUtils.getMachinesFromAgentUids(pu, aggregatedAllocatedCapacity.getAgentUids());
        if (!RebalancingUtils.isProcessingUnitIntact(pu, machines)) {
            isEvenlyDistributedAcrossMachines = false;
        } else {
            Fraction averageCpuCoresPerPrimaryInstance = RebalancingUtils.getAverageCpuCoresPerPrimary(pu, aggregatedAllocatedCapacity);
            block0: for (Machine source : machines) {
                for (Machine target : machines) {
                    if (target.equals(source) || !RebalancingUtils.isRestartRecommended(pu, source, target, averageCpuCoresPerPrimaryInstance, aggregatedAllocatedCapacity)) continue;
                    isEvenlyDistributedAcrossMachines = false;
                    continue block0;
                }
            }
        }
        return isEvenlyDistributedAcrossMachines;
    }

    private static Machine[] getMachinesFromAgentUids(ProcessingUnit pu, Collection<String> agentUids) {
        ArrayList<Machine> machines = new ArrayList<Machine>();
        GridServiceAgents gridServiceAgents = pu.getAdmin().getGridServiceAgents();
        for (String agentUid : agentUids) {
            GridServiceAgent agent = gridServiceAgents.getAgentByUID(agentUid);
            if (agent == null) {
                throw new IllegalStateException("At this point agent " + agentUid + " must be discovered.");
            }
            machines.add(agent.getMachine());
        }
        return machines.toArray(new Machine[machines.size()]);
    }

    public static boolean isRestartRecommended(ProcessingUnit pu, Machine source, Machine target, Fraction optimalCpuCoresPerPrimary, CapacityRequirementsPerAgent allocatedCapacity) {
        boolean isRestartRecommended = false;
        int numberOfPrimaryInstancesOnSource = RebalancingUtils.getNumberOfPrimaryInstancesOnMachine(pu, source);
        if (numberOfPrimaryInstancesOnSource > 0) {
            int numberOfPrimaryInstancesOnTarget = RebalancingUtils.getNumberOfPrimaryInstancesOnMachine(pu, target);
            Fraction cpuCoresOnSource = RebalancingUtils.getNumberOfCpuCores(source, allocatedCapacity);
            Fraction cpuCoresOnTarget = RebalancingUtils.getNumberOfCpuCores(target, allocatedCapacity);
            Fraction missingCpuCoresBeforeRestart = RebalancingUtils.max(Fraction.ZERO, optimalCpuCoresPerPrimary.multiply(numberOfPrimaryInstancesOnSource).subtract(cpuCoresOnSource)).add(RebalancingUtils.max(Fraction.ZERO, optimalCpuCoresPerPrimary.multiply(numberOfPrimaryInstancesOnTarget).subtract(cpuCoresOnTarget)));
            Fraction missingCpuCoresAfterRestart = RebalancingUtils.max(Fraction.ZERO, optimalCpuCoresPerPrimary.multiply(numberOfPrimaryInstancesOnSource - 1).subtract(cpuCoresOnSource)).add(RebalancingUtils.max(Fraction.ZERO, optimalCpuCoresPerPrimary.multiply(numberOfPrimaryInstancesOnTarget + 1).subtract(cpuCoresOnTarget)));
            isRestartRecommended = missingCpuCoresAfterRestart.compareTo(missingCpuCoresBeforeRestart) < 0;
        }
        return isRestartRecommended;
    }

    private static Fraction max(Fraction a, Fraction b) {
        if (b.compareTo(a) > 0) {
            return b;
        }
        return a;
    }

    private static boolean isProcessingUnitIntact(ProcessingUnit pu, Machine[] machines) {
        return RebalancingUtils.isProcessingUnitIntact(pu, RebalancingUtils.getContainersOnMachines(pu, machines));
    }

    public static GridServiceContainer[] getContainersOnMachines(ProcessingUnit pu) {
        return RebalancingUtils.getContainersOnMachines(pu, pu.getAdmin().getMachines().getMachines());
    }

    private static GridServiceContainer[] getContainersOnMachines(ProcessingUnit pu, Machine[] machines) {
        if (pu.getRequiredZones().length != 1) {
            throw new IllegalStateException("Processing Unit must have exactly one container zone defined.");
        }
        ArrayList<GridServiceContainer> containers = new ArrayList<GridServiceContainer>();
        for (Machine machine : machines) {
            for (GridServiceContainer container : machine.getGridServiceContainers()) {
                if (container.getZones().size() != 1 || !container.getZones().containsKey(pu.getRequiredZones()[0])) continue;
                containers.add(container);
            }
        }
        return containers.toArray(new GridServiceContainer[containers.size()]);
    }

    public static boolean isEvenlyDistributedAcrossContainers(ProcessingUnit pu, GridServiceContainer[] containers) {
        int numberOfContainers;
        if (!RebalancingUtils.isProcessingUnitIntact(pu, containers)) {
            return false;
        }
        boolean evenlyDistributed = true;
        int numberOfInstances = pu.getTotalNumberOfInstances();
        if (numberOfInstances < (numberOfContainers = containers.length)) {
            evenlyDistributed = false;
        } else {
            double expectedAverageNumberOfInstancesPerContainer = 1.0 * (double)numberOfInstances / (double)numberOfContainers;
            int numberOfServicesPerContainerUpperBound = (int)Math.ceil(expectedAverageNumberOfInstancesPerContainer);
            int numberOfServicesPerContainerLowerBound = (int)Math.floor(expectedAverageNumberOfInstancesPerContainer);
            for (GridServiceContainer container : containers) {
                int puNumberOfInstances = container.getProcessingUnitInstances(pu.getName()).length;
                if (puNumberOfInstances >= numberOfServicesPerContainerLowerBound && puNumberOfInstances <= numberOfServicesPerContainerUpperBound) continue;
                evenlyDistributed = false;
                break;
            }
        }
        return evenlyDistributed;
    }

    public static Machine[] getMachinesHostingContainers(GridServiceContainer[] containers) {
        HashSet<Machine> machines = new HashSet<Machine>();
        for (GridServiceContainer container : containers) {
            machines.add(container.getMachine());
        }
        return machines.toArray(new Machine[machines.size()]);
    }

    public static int getPlannedMinimumNumberOfInstancesForContainer(GridServiceContainer container, GridServiceContainer[] approvedContainers, ProcessingUnit pu) {
        int min = 0;
        if (Arrays.asList(approvedContainers).contains(container)) {
            min = (int)Math.floor(RebalancingUtils.getAverageNumberOfInstancesPerContainer(approvedContainers, pu));
        }
        return min;
    }

    public static int getPlannedMaximumNumberOfInstancesForContainer(GridServiceContainer container, GridServiceContainer[] approvedContainers, ProcessingUnit pu) {
        int max = 0;
        if (Arrays.asList(approvedContainers).contains(container)) {
            max = (int)Math.ceil(RebalancingUtils.getAverageNumberOfInstancesPerContainer(approvedContainers, pu));
        }
        return max;
    }

    private static double getAverageNumberOfInstancesPerContainer(GridServiceContainer[] approvedContainers, ProcessingUnit pu) {
        double avg = (double)pu.getTotalNumberOfInstances() / (double)approvedContainers.length;
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("averageInstancesPerContainer = ((double) pu.getTotalNumberOfInstances()) / approvedContainers.length = " + (double)pu.getTotalNumberOfInstances() + "/" + approvedContainers.length + " = " + avg));
        }
        return avg;
    }

    public static List<GridServiceContainer> sortAllContainersByNumberOfInstancesAboveMinimum(final ProcessingUnit pu, final GridServiceContainer[] approvedContainers) {
        ArrayList<GridServiceContainer> sortedContainers = new ArrayList<GridServiceContainer>(Arrays.asList(pu.getAdmin().getGridServiceContainers().getContainers()));
        Collections.sort(sortedContainers, new Comparator<GridServiceContainer>(){

            @Override
            public int compare(GridServiceContainer o1, GridServiceContainer o2) {
                return this.getNormalizedNumberOfInstances(o1) - this.getNormalizedNumberOfInstances(o2);
            }

            private int getNormalizedNumberOfInstances(GridServiceContainer container) {
                int numberOfInstances = container.getProcessingUnitInstances(pu.getName()).length;
                return numberOfInstances - RebalancingUtils.getPlannedMinimumNumberOfInstancesForContainer(container, approvedContainers, pu);
            }
        });
        return sortedContainers;
    }

    public static List<Machine> sortMachinesByNumberOfPrimaryInstancesPerCpuCore(final ProcessingUnit pu, Machine[] machines, final CapacityRequirementsPerAgent allocatedCapacity) {
        ArrayList<Machine> sortedMachines = new ArrayList<Machine>(Arrays.asList(machines));
        Collections.sort(sortedMachines, new Comparator<Machine>(){

            @Override
            public int compare(Machine m1, Machine m2) {
                if (RebalancingUtils.getNumberOfCpuCores(m1, allocatedCapacity).equals((Object)Fraction.ZERO) || RebalancingUtils.getNumberOfCpuCores(m2, allocatedCapacity).equals((Object)Fraction.ZERO)) {
                    throw new IllegalStateException("Rebalancing assumes positive number of CPU cores per machine");
                }
                return RebalancingUtils.getNumberOfPrimaryInstancesPerCpuCore(pu, m1, allocatedCapacity).compareTo(RebalancingUtils.getNumberOfPrimaryInstancesPerCpuCore(pu, m2, allocatedCapacity));
            }
        });
        return sortedMachines;
    }

    public static Fraction getNumberOfPrimaryInstancesPerCpuCore(ProcessingUnit pu, Machine machine, CapacityRequirementsPerAgent allocatedCapacity) {
        return new Fraction(RebalancingUtils.getNumberOfPrimaryInstancesOnMachine(pu, machine)).divide(RebalancingUtils.getNumberOfCpuCores(machine, allocatedCapacity));
    }

    public static int getNumberOfPrimaryInstancesOnMachine(ProcessingUnit pu, Machine machine) {
        int numberOfPrimaryInstances = 0;
        for (GridServiceContainer container : machine.getGridServiceContainers()) {
            for (ProcessingUnitInstance instance : container.getProcessingUnitInstances(pu.getName())) {
                if (instance.getSpaceInstance() == null || instance.getSpaceInstance().getMode() != SpaceMode.PRIMARY) continue;
                ++numberOfPrimaryInstances;
            }
        }
        return numberOfPrimaryInstances;
    }

    public static FutureStatefulProcessingUnitInstance restartProcessingUnitInstanceAsync(ProcessingUnitInstance candidateInstance, Log logger, long timeout, TimeUnit timeUnit) {
        return RebalancingUtils.relocateProcessingUnitInstanceAsync(candidateInstance.getGridServiceContainer(), candidateInstance, logger, timeout, timeUnit);
    }

    public static Fraction getAverageCpuCoresPerPrimary(ProcessingUnit pu, CapacityRequirementsPerAgent aggregatedAllocatedCapacity) {
        CapacityRequirements totalAllocatedCapacity = aggregatedAllocatedCapacity.getTotalAllocatedCapacity();
        if (totalAllocatedCapacity.equalsZero()) {
            throw new IllegalStateException("allocated capacity cannot be empty.");
        }
        return RebalancingUtils.getCpuCores(totalAllocatedCapacity).divide(pu.getNumberOfInstances());
    }

    private static Fraction getCpuCores(CapacityRequirements totalAllocatedCapacity) {
        return totalAllocatedCapacity.getRequirement(new CpuCapacityRequirement().getType()).getCpu();
    }

    public static Fraction getNumberOfCpuCores(Machine machine, CapacityRequirementsPerAgent allocatedCapacity) {
        if (machine.getGridServiceAgents().getSize() != 1) {
            throw new IllegalStateException("Machine must have at least one agent");
        }
        return RebalancingUtils.getCpuCores(allocatedCapacity.getAgentCapacity(machine.getGridServiceAgent().getUid()));
    }

    public static String puInstanceToString(ProcessingUnitInstance instance) {
        StringBuilder builder = new StringBuilder(16);
        builder.append("[").append(instance.getInstanceId()).append(",").append(instance.getBackupId() + 1);
        SpaceInstance spaceInstance = instance.getSpaceInstance();
        if (spaceInstance != null) {
            builder.append(",").append(spaceInstance.getMode());
        }
        builder.append("]");
        return builder.toString();
    }

    public static String machineToString(Machine machine) {
        return machine.getHostName() + "/" + machine.getHostAddress();
    }

    public static String gscToString(GridComponent container) {
        return ContainersSlaUtils.gscToString(container);
    }

    public static String gscsToString(List<GridServiceContainer> containers) {
        return ContainersSlaUtils.gscsToString(containers);
    }

    public static String processingUnitDeploymentToString(ProcessingUnit pu) {
        StringBuilder deployment = new StringBuilder();
        for (GridServiceContainer container : pu.getAdmin().getGridServiceContainers()) {
            deployment.append(RebalancingUtils.gscToString(container));
            deployment.append(" { ");
            for (ProcessingUnitInstance instance : container.getProcessingUnitInstances(pu.getName())) {
                deployment.append(RebalancingUtils.puInstanceToString(instance));
                deployment.append(" ");
            }
            deployment.append(" } ");
        }
        return deployment.toString();
    }
}

