/*
 * Decompiled with CFR 0.152.
 */
package com.gigaspaces.internal.license;

import com.gigaspaces.internal.license.LicenseType;
import com.gigaspaces.internal.utils.StringUtils;
import com.gigaspaces.internal.utils.concurrent.GSThreadFactory;
import com.gigaspaces.internal.version.PlatformVersion;
import com.gigaspaces.license.LicenseException;
import com.gigaspaces.start.Locator;
import com.gigaspaces.start.ProductType;
import com.gigaspaces.start.SystemInfo;
import com.j_spaces.kernel.ResourceLoader;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Security;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.Properties;
import java.util.Scanner;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import sun.security.provider.Sun;

public class LicenseManager {
    private static Logger logger = Logger.getLogger("com.gigaspaces.license");
    private static final String UNBOUND = "UNBOUND";
    private static final String UNLIMITED = "UNLIMITED";
    private static final String TRIAL_KEY = "tryme";
    private static final String PRODUCT_PROP = "Product";
    private static final String VERSION_PROP = "Version";
    private static final String TYPE_PROP = "Type";
    private static final String CUSTOMER_PROP = "Customer";
    private static final String EXPIRATION_PROP = "Expiration";
    private static final int EVALUATION_DAY_LIMIT = 30;
    private static LicenseManager instance;
    private final ProductType currProductType;
    private final String currVersion;
    private final String key;
    private final boolean trial;
    private final String product;
    private final String version;
    private final LicenseType type;
    private final String user;
    private final Date expirationDate;
    private final String host;
    private final Integer cores;

    public static synchronized LicenseManager getInstance() {
        if (instance == null) {
            instance = new LicenseManager(LicenseManager.loadLicenseKey(), PlatformVersion.getInstance());
        }
        return instance;
    }

    public static void setLicenseKey(String licenseKey) {
        System.setProperty("com.gs.licensekey", licenseKey);
    }

    public static String getLicenseKeyWithoutValidation() {
        return LicenseManager.loadLicenseKey();
    }

    public LicenseManager(String key, PlatformVersion platformVersion) {
        this.key = key;
        this.currVersion = platformVersion.getMajorVersion() + "." + platformVersion.getMinorVersion();
        this.currProductType = platformVersion.getProductType();
        this.trial = TRIAL_KEY.equalsIgnoreCase(key);
        Properties properties = this.trial ? LicenseManager.generateTrialLicense(this.currProductType, this.currVersion, TimeUnit.DAYS.toMillis(1L)) : StringUtils.toProperties((String)LicenseManager.validateHash(key), (String)";", (String)"=");
        this.product = properties.getProperty(PRODUCT_PROP);
        this.version = properties.getProperty(VERSION_PROP);
        this.type = LicenseType.fromString(properties.getProperty(TYPE_PROP));
        this.user = properties.getProperty(CUSTOMER_PROP);
        this.expirationDate = this.toDate(properties.getProperty(EXPIRATION_PROP, UNLIMITED));
        this.host = properties.getProperty("Host", UNBOUND);
        this.cores = LicenseManager.toInt(properties.getProperty("Cores", UNLIMITED));
        if (this.trial) {
            new LicenseExpirationMonitor(this.expirationDate.getTime(), TimeUnit.MINUTES.toMillis(5L), TimeUnit.SECONDS.toMillis(10L));
        }
    }

    private static Properties generateTrialLicense(ProductType currProductType, String currVersion, long trialPeriod) {
        Properties p = new Properties();
        p.setProperty(PRODUCT_PROP, currProductType.name());
        p.setProperty(VERSION_PROP, currVersion);
        p.setProperty(TYPE_PROP, LicenseType.ENTERPRISE.name());
        p.setProperty(CUSTOMER_PROP, TRIAL_KEY);
        p.setProperty(EXPIRATION_PROP, LicenseManager.getDataFormat(true).format(new Date(System.currentTimeMillis() + trialPeriod)));
        return p;
    }

    public String getLicenseKey() {
        return this.key;
    }

    public LicenseType getLicenseType() {
        return this.type;
    }

    public Date getExpirationDate() {
        return this.expirationDate;
    }

    public String getUser() {
        return this.user;
    }

    public void validate(String featureName, LicenseType requiredLicenseType) {
        try {
            this.validateProduct();
            this.validateVersion();
            this.validateExpiration();
            this.validateHost();
            this.validateProcessors();
            if (!this.type.includes(requiredLicenseType)) {
                throw LicenseException.invalid(this.key, "Using " + featureName + " requires " + requiredLicenseType.name() + " license. Please contact support for more details: http://www.gigaspaces.com/supportcenter");
            }
        }
        catch (LicenseException e) {
            logger.severe(e.getMessage());
            throw e;
        }
    }

    private void validateProduct() {
        if (this.product != null && this.currProductType == ProductType.InsightEdge && this.product.equals(ProductType.XAP.name())) {
            throw LicenseException.invalid(this.key, "This license was issued for [" + this.product + "] whereas the current product is [" + this.currProductType + "]");
        }
    }

    private void validateVersion() {
        if (this.version != null && !this.version.equals(this.currVersion)) {
            throw LicenseException.invalid(this.key, "This license was issued for version [" + this.version + "] whereas the current version is [" + this.currVersion + "]");
        }
    }

    private void validateExpiration() {
        if (this.expirationDate != null) {
            long currTime = System.currentTimeMillis();
            if (this.expirationDate.getTime() < currTime) {
                throw LicenseException.invalid(this.key, "License has expired");
            }
            long difference = Math.abs(this.expirationDate.getTime() - currTime);
            long totalDays = difference / 86400000L + 1L;
            if (totalDays <= 30L && logger.isLoggable(Level.INFO)) {
                logger.info("\n-----------------------------------------------------------------------------------\n*  Your license will expire in " + totalDays + " days.\n*  To extend your evaluation period please contact your local GigaSpaces(TM) \n*  software sales representative or access GigaSpaces Support Center at \n*  www.gigaspaces.com\n-----------------------------------------------------------------------------------");
            }
        }
    }

    private void validateHost() {
        String currentHost;
        if (!this.host.equalsIgnoreCase(UNBOUND) && !this.host.equalsIgnoreCase(currentHost = SystemInfo.singleton().network().getHost().getHostName())) {
            throw LicenseException.invalid(this.key, "License is bound to host " + this.host + "; current host is: " + currentHost);
        }
    }

    private void validateProcessors() {
        int currCores;
        if (this.cores != null && (currCores = Runtime.getRuntime().availableProcessors()) > this.cores) {
            throw LicenseException.invalid(this.key, "This license permits " + this.cores + " cores, but the number of cores detected on this machine is " + currCores);
        }
    }

    private static String loadLicenseKey() {
        String fileName = "xap-license.txt";
        String licenseKey = LicenseManager.getLicenseFromSystemProperty("com.gs.licensekey");
        if (licenseKey == null) {
            licenseKey = LicenseManager.getLicenseFromEnvironmentVariable("XAP_LICENSE");
        }
        if (licenseKey == null) {
            licenseKey = LicenseManager.getLicenseFromClasspath("xap-license.txt");
        }
        if (licenseKey == null) {
            licenseKey = LicenseManager.getLicenseFromFile(SystemInfo.singleton().getXapHome() + File.separator + "xap-license.txt");
        }
        if (licenseKey == null) {
            licenseKey = LicenseManager.getLicenseFromFile("." + File.separator + "xap-license.txt");
        }
        if (licenseKey != null) {
            int beginIndex = licenseKey.startsWith("\"") ? 1 : 0;
            int endIndex = licenseKey.endsWith("\"") ? licenseKey.length() - 1 : licenseKey.length();
            licenseKey = licenseKey.substring(beginIndex, endIndex);
            LicenseManager.setLicenseKey(licenseKey);
        }
        return licenseKey;
    }

    public static String getLicenseFromFile(String fileName) {
        File file = new File(fileName);
        if (!file.exists()) {
            return null;
        }
        try {
            Scanner scanner = new Scanner(file);
            String licenseKey = LicenseManager.readFirstLine(scanner);
            if (licenseKey != null && logger.isLoggable(Level.CONFIG)) {
                logger.config("License loaded from file '" + fileName + "' - " + licenseKey);
            }
            return licenseKey;
        }
        catch (FileNotFoundException e) {
            return null;
        }
    }

    private static String getLicenseFromClasspath(String fileName) {
        String gsBootLib = (String)Locator.deriveDirectories().get("com.gigaspaces.lib.boot");
        InputStream inputStream = ResourceLoader.getResourceStream((String)fileName, (String)gsBootLib);
        if (inputStream == null) {
            return null;
        }
        String licenseKey = LicenseManager.readFirstLine(new Scanner(inputStream));
        if (licenseKey != null && logger.isLoggable(Level.CONFIG)) {
            logger.config("License loaded from class path '" + ResourceLoader.getResourceURL((String)fileName, (String)gsBootLib) + "' - " + licenseKey);
        }
        LicenseManager.safeClose(inputStream, logger);
        return licenseKey;
    }

    private static String readFirstLine(Scanner scanner) {
        while (scanner.hasNextLine()) {
            String s = scanner.nextLine().trim();
            if (s.length() == 0 || s.startsWith("#")) continue;
            return s;
        }
        scanner.close();
        return null;
    }

    private static void safeClose(Closeable resource, Logger logger) {
        block3: {
            if (resource != null) {
                try {
                    resource.close();
                }
                catch (IOException e) {
                    if (!logger.isLoggable(Level.WARNING)) break block3;
                    logger.log(Level.WARNING, "Failed to close", e);
                }
            }
        }
    }

    private static String getLicenseFromSystemProperty(String systemProperty) {
        String licenseKey = System.getProperty(systemProperty);
        if (licenseKey != null && logger.isLoggable(Level.CONFIG)) {
            logger.config("License taken from system property '" + systemProperty + "' - " + licenseKey);
        }
        return licenseKey;
    }

    private static String getLicenseFromEnvironmentVariable(String key) {
        String licenseKey = System.getenv(key);
        if (licenseKey != null && logger.isLoggable(Level.CONFIG)) {
            logger.config("License taken from environment variable '" + key + "' - " + licenseKey);
        }
        return licenseKey;
    }

    private Date toDate(String s) {
        try {
            return s.equals(UNLIMITED) ? null : LicenseManager.getDataFormat(s.contains(" ")).parse(s);
        }
        catch (ParseException e) {
            throw LicenseException.invalid(this.key, "Failed to parse expiration date - " + e.getMessage());
        }
    }

    private static DateFormat getDataFormat(boolean includeTime) {
        String pattern = includeTime ? "yyyy-MMM-dd HH:mm:ss" : "yyyy-MMM-dd";
        return new SimpleDateFormat(pattern, Locale.US);
    }

    private static Integer toInt(String s) {
        return s.equals(UNLIMITED) ? null : Integer.valueOf(Integer.parseInt(s));
    }

    private static String validateHash(String key) {
        if (key == null || key.length() == 0) {
            throw new LicenseException("License was not specified. Please contact support for more details.");
        }
        String SUFFIX = ";Hash=";
        int hashPos = key.indexOf(";Hash=");
        if (hashPos == -1) {
            throw LicenseException.invalid(key, "License data is corrupted");
        }
        String hash = key.substring(hashPos + ";Hash=".length());
        String licenseWithoutHash = key.substring(0, hashPos);
        MessageDigest messageDigest = LicenseManager.initMessageDigest();
        messageDigest.reset();
        if (!MessageDigest.isEqual(hash.getBytes(), LicenseManager.byteToAscii(messageDigest.digest(licenseWithoutHash.getBytes())))) {
            throw LicenseException.invalid(key, "License data is corrupted");
        }
        return licenseWithoutHash;
    }

    private static MessageDigest initMessageDigest() {
        block4: {
            try {
                Security.addProvider(new Sun());
            }
            catch (SecurityException e) {
                if (!logger.isLoggable(Level.SEVERE)) break block4;
                logger.log(Level.SEVERE, "Failed to add the sun.security.provider.Sun SecurityManager: " + e.getMessage(), e);
            }
        }
        try {
            return MessageDigest.getInstance("SHA");
        }
        catch (NoSuchAlgorithmException e) {
            throw new LicenseException("Failed to create LicenseManager", e);
        }
    }

    private static byte[] byteToAscii(byte[] data) {
        int len = data.length;
        byte[] result = new byte[len];
        double d = 0.09615384615384616;
        for (int i = 0; i < len; ++i) {
            int intVal = Math.abs(data[i]);
            byte type = (byte)Character.getType((char)intVal);
            if (type != 9 && type != 1 && type != 2 && type != 10) {
                intVal = (int)(d * (double)intVal + 90.0 - 12.211538461538462);
            }
            result[i] = (byte)intVal;
        }
        return result;
    }

    private static class LicenseExpirationMonitor {
        private static final String EVAL_URL = "gigaspaces.com/eval-license";
        private final ScheduledExecutorService scheduler;
        private final long expiration;
        private final long monitorInterval;
        private final long aboutToExpireDuration;

        private LicenseExpirationMonitor(long expiration, long monitorInterval, long aboutToExpireDuration) {
            this.expiration = expiration;
            this.monitorInterval = monitorInterval;
            this.aboutToExpireDuration = aboutToExpireDuration;
            this.scheduler = Executors.newScheduledThreadPool(1, (ThreadFactory)new GSThreadFactory("LicenseMonitor", true));
            logger.info("You are using the GigaSpaces '" + LicenseManager.TRIAL_KEY.toUpperCase() + "' trial license. You can explore all of the product functionality, including our Premium and Enterprise features, for 24 hours.");
            logger.info("When the trial license expires, this process will self-terminate. You can get an extended evaluation license from gigaspaces.com/eval-license.");
            this.monitorTrialExpiration();
        }

        private void monitorTrialExpiration() {
            long remaining = this.expiration - System.currentTimeMillis();
            if (remaining > this.aboutToExpireDuration) {
                long interval = remaining - this.aboutToExpireDuration > this.monitorInterval ? this.monitorInterval : remaining - this.aboutToExpireDuration;
                logger.info("Remaining time before your trial license expires and process terminates: " + this.formatDuration(remaining));
                this.scheduler.schedule(new Runnable(){

                    @Override
                    public void run() {
                        this.monitorTrialExpiration();
                    }
                }, interval, TimeUnit.MILLISECONDS);
            } else {
                try {
                    long r;
                    while ((r = this.expiration - System.currentTimeMillis()) > 1000L) {
                        logger.info("Your trial license is about to expire. Time until process termination: " + this.formatDuration(r));
                        Thread.sleep(1000L);
                    }
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                logger.info("Your trial license has expired, the process will now terminate.");
                logger.info("You can get an extended evaluation license from gigaspaces.com/eval-license.");
                System.exit(0);
            }
        }

        private String formatDuration(long duration) {
            long s = duration / 1000L;
            return String.format("%d:%02d:%02d", s / 3600L, s % 3600L / 60L, s % 60L);
        }
    }
}

