/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.notebook.repo;

import com.google.common.collect.Lists;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.NoteInfo;
import org.apache.zeppelin.notebook.NotebookAuthorization;
import org.apache.zeppelin.notebook.Paragraph;
import org.apache.zeppelin.notebook.repo.NotebookRepo;
import org.apache.zeppelin.notebook.repo.NotebookRepoSettingsInfo;
import org.apache.zeppelin.notebook.repo.NotebookRepoWithSettings;
import org.apache.zeppelin.notebook.repo.NotebookRepoWithVersionControl;
import org.apache.zeppelin.user.AuthenticationInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NotebookRepoSync
implements NotebookRepoWithVersionControl {
    private static final Logger LOG = LoggerFactory.getLogger(NotebookRepoSync.class);
    private static final int maxRepoNum = 2;
    private static final String pushKey = "pushNoteIds";
    private static final String pullKey = "pullNoteIds";
    private static final String delDstKey = "delDstNoteIds";
    private static ZeppelinConfiguration config;
    private static final String defaultStorage = "org.apache.zeppelin.notebook.repo.GitNotebookRepo";
    private List<NotebookRepo> repos = new ArrayList<NotebookRepo>();
    private final boolean oneWaySync;

    public NotebookRepoSync(ZeppelinConfiguration conf) {
        String[] storageClassNames;
        config = conf;
        this.oneWaySync = conf.getBoolean(ZeppelinConfiguration.ConfVars.ZEPPELIN_NOTEBOOK_ONE_WAY_SYNC);
        String allStorageClassNames = conf.getString(ZeppelinConfiguration.ConfVars.ZEPPELIN_NOTEBOOK_STORAGE).trim();
        if (allStorageClassNames.isEmpty()) {
            allStorageClassNames = defaultStorage;
            LOG.warn("Empty ZEPPELIN_NOTEBOOK_STORAGE conf parameter, using default {}", (Object)defaultStorage);
        }
        if ((storageClassNames = allStorageClassNames.split(",")).length > this.getMaxRepoNum()) {
            LOG.warn("Unsupported number {} of storage classes in ZEPPELIN_NOTEBOOK_STORAGE : {}\nfirst {} will be used", new Object[]{storageClassNames.length, allStorageClassNames, this.getMaxRepoNum()});
        }
        for (int i = 0; i < Math.min(storageClassNames.length, this.getMaxRepoNum()); ++i) {
            try {
                this.getClass();
                Class<?> notebookStorageClass = Class.forName(storageClassNames[i].trim());
                Constructor<?> constructor = notebookStorageClass.getConstructor(ZeppelinConfiguration.class);
                this.repos.add((NotebookRepo)constructor.newInstance(conf));
                LOG.info("Instantiate NotebookRepo: " + storageClassNames[i]);
                continue;
            }
            catch (ClassNotFoundException | IllegalAccessException | IllegalArgumentException | InstantiationException | NoSuchMethodException | SecurityException | InvocationTargetException e) {
                LOG.warn("Failed to initialize {} notebook storage class", (Object)storageClassNames[i], (Object)e);
            }
        }
        if (this.getRepoCount() == 0) {
            LOG.info("No storage could be initialized, using default {} storage", (Object)defaultStorage);
            this.initializeDefaultStorage(conf);
        }
        if (this.getRepoCount() > 1 && conf.getBoolean(ZeppelinConfiguration.ConfVars.ZEPPELIN_ANONYMOUS_ALLOWED)) {
            try {
                this.sync(AuthenticationInfo.ANONYMOUS);
            }
            catch (IOException e) {
                LOG.error("Couldn't sync on start ", (Throwable)e);
            }
        }
    }

    private void initializeDefaultStorage(ZeppelinConfiguration conf) {
        try {
            this.getClass();
            Class<?> notebookStorageClass = Class.forName(defaultStorage);
            Constructor<?> constructor = notebookStorageClass.getConstructor(ZeppelinConfiguration.class);
            this.repos.add((NotebookRepo)constructor.newInstance(conf));
        }
        catch (ClassNotFoundException | IllegalAccessException | IllegalArgumentException | InstantiationException | NoSuchMethodException | SecurityException | InvocationTargetException e) {
            LOG.warn("Failed to initialize {} notebook storage class {}", (Object)defaultStorage, (Object)e);
        }
    }

    public List<NotebookRepoWithSettings> getNotebookRepos(AuthenticationInfo subject) {
        ArrayList reposSetting = Lists.newArrayList();
        for (NotebookRepo repo : this.repos) {
            NotebookRepoWithSettings repoWithSettings = NotebookRepoWithSettings.builder(repo.getClass().getSimpleName()).className(repo.getClass().getName()).settings(repo.getSettings(subject)).build();
            reposSetting.add(repoWithSettings);
        }
        return reposSetting;
    }

    public NotebookRepoWithSettings updateNotebookRepo(String name, Map<String, String> settings, AuthenticationInfo subject) {
        NotebookRepoWithSettings updatedSettings = NotebookRepoWithSettings.EMPTY;
        for (NotebookRepo repo : this.repos) {
            if (!repo.getClass().getName().equals(name)) continue;
            repo.updateSettings(settings, subject);
            updatedSettings = NotebookRepoWithSettings.builder(repo.getClass().getSimpleName()).className(repo.getClass().getName()).settings(repo.getSettings(subject)).build();
            break;
        }
        return updatedSettings;
    }

    @Override
    public List<NoteInfo> list(AuthenticationInfo subject) throws IOException {
        return this.getRepo(0).list(subject);
    }

    List<NoteInfo> list(int repoIndex, AuthenticationInfo subject) throws IOException {
        return this.getRepo(repoIndex).list(subject);
    }

    @Override
    public Note get(String noteId, AuthenticationInfo subject) throws IOException {
        return this.getRepo(0).get(noteId, subject);
    }

    Note get(int repoIndex, String noteId, AuthenticationInfo subject) throws IOException {
        return this.getRepo(repoIndex).get(noteId, subject);
    }

    @Override
    public void save(Note note, AuthenticationInfo subject) throws IOException {
        this.getRepo(0).save(note, subject);
        if (this.getRepoCount() > 1) {
            try {
                this.getRepo(1).save(note, subject);
            }
            catch (IOException e) {
                LOG.info(e.getMessage() + ": Failed to write to secondary storage");
            }
        }
    }

    void save(int repoIndex, Note note, AuthenticationInfo subject) throws IOException {
        this.getRepo(repoIndex).save(note, subject);
    }

    @Override
    public void remove(String noteId, AuthenticationInfo subject) throws IOException {
        for (NotebookRepo repo : this.repos) {
            repo.remove(noteId, subject);
        }
    }

    void remove(int repoIndex, String noteId, AuthenticationInfo subject) throws IOException {
        this.getRepo(repoIndex).remove(noteId, subject);
    }

    void sync(int sourceRepoIndex, int destRepoIndex, AuthenticationInfo subject) throws IOException {
        LOG.info("Sync started");
        NotebookAuthorization auth = NotebookAuthorization.getInstance();
        NotebookRepo srcRepo = this.getRepo(sourceRepoIndex);
        NotebookRepo dstRepo = this.getRepo(destRepoIndex);
        List<NoteInfo> allSrcNotes = srcRepo.list(subject);
        List<NoteInfo> srcNotes = auth.filterByUser(allSrcNotes, subject);
        List<NoteInfo> dstNotes = dstRepo.list(subject);
        Map<String, List<String>> noteIds = this.notesCheckDiff(srcNotes, srcRepo, dstNotes, dstRepo, subject);
        List<String> pushNoteIds = noteIds.get(pushKey);
        List<String> pullNoteIds = noteIds.get(pullKey);
        List<String> delDstNoteIds = noteIds.get(delDstKey);
        if (!pushNoteIds.isEmpty()) {
            LOG.info("Notes with the following IDs will be pushed");
            for (String id : pushNoteIds) {
                LOG.info("ID : " + id);
            }
            this.pushNotes(subject, pushNoteIds, srcRepo, dstRepo, false);
        } else {
            LOG.info("Nothing to push");
        }
        if (!pullNoteIds.isEmpty()) {
            LOG.info("Notes with the following IDs will be pulled");
            for (String id : pullNoteIds) {
                LOG.info("ID : " + id);
            }
            this.pushNotes(subject, pullNoteIds, dstRepo, srcRepo, true);
        } else {
            LOG.info("Nothing to pull");
        }
        if (!delDstNoteIds.isEmpty()) {
            LOG.info("Notes with the following IDs will be deleted from dest");
            for (String id : delDstNoteIds) {
                LOG.info("ID : " + id);
            }
            this.deleteNotes(subject, delDstNoteIds, dstRepo);
        } else {
            LOG.info("Nothing to delete from dest");
        }
        LOG.info("Sync ended");
    }

    public void sync(AuthenticationInfo subject) throws IOException {
        this.sync(0, 1, subject);
    }

    private void pushNotes(AuthenticationInfo subject, List<String> ids, NotebookRepo localRepo, NotebookRepo remoteRepo, boolean setPermissions) {
        for (String id : ids) {
            try {
                remoteRepo.save(localRepo.get(id, subject), subject);
                if (!setPermissions || !this.emptyNoteAcl(id)) continue;
                this.makePrivate(id, subject);
            }
            catch (IOException e) {
                LOG.error("Failed to push note to storage, moving onto next one", (Throwable)e);
            }
        }
    }

    private boolean emptyNoteAcl(String noteId) {
        NotebookAuthorization notebookAuthorization = NotebookAuthorization.getInstance();
        return notebookAuthorization.getOwners(noteId).isEmpty() && notebookAuthorization.getReaders(noteId).isEmpty() && notebookAuthorization.getRunners(noteId).isEmpty() && notebookAuthorization.getWriters(noteId).isEmpty();
    }

    private void makePrivate(String noteId, AuthenticationInfo subject) {
        if (AuthenticationInfo.isAnonymous((AuthenticationInfo)subject)) {
            LOG.info("User is anonymous, permissions are not set for pulled notes");
            return;
        }
        NotebookAuthorization notebookAuthorization = NotebookAuthorization.getInstance();
        Set<String> users = notebookAuthorization.getOwners(noteId);
        users.add(subject.getUser());
        notebookAuthorization.setOwners(noteId, users);
        users = notebookAuthorization.getReaders(noteId);
        users.add(subject.getUser());
        notebookAuthorization.setReaders(noteId, users);
        users = notebookAuthorization.getRunners(noteId);
        users.add(subject.getUser());
        notebookAuthorization.setRunners(noteId, users);
        users = notebookAuthorization.getWriters(noteId);
        users.add(subject.getUser());
        notebookAuthorization.setWriters(noteId, users);
    }

    private void deleteNotes(AuthenticationInfo subject, List<String> ids, NotebookRepo repo) throws IOException {
        for (String id : ids) {
            repo.remove(id, subject);
        }
    }

    public int getRepoCount() {
        return this.repos.size();
    }

    int getMaxRepoNum() {
        return 2;
    }

    public NotebookRepo getRepo(int repoIndex) throws IOException {
        if (repoIndex < 0 || repoIndex >= this.getRepoCount()) {
            throw new IOException("Requested storage index " + repoIndex + " isn't initialized, repository count is " + this.getRepoCount());
        }
        return this.repos.get(repoIndex);
    }

    private Map<String, List<String>> notesCheckDiff(List<NoteInfo> sourceNotes, NotebookRepo sourceRepo, List<NoteInfo> destNotes, NotebookRepo destRepo, AuthenticationInfo subject) {
        NoteInfo dnote;
        ArrayList<String> pushIDs = new ArrayList<String>();
        ArrayList<String> pullIDs = new ArrayList<String>();
        ArrayList<String> delDstIDs = new ArrayList<String>();
        for (NoteInfo snote : sourceNotes) {
            dnote = this.containsID(destNotes, snote.getId());
            if (dnote != null) {
                Date ddate;
                Date sdate;
                try {
                    sdate = this.lastModificationDate(sourceRepo.get(snote.getId(), subject));
                    ddate = this.lastModificationDate(destRepo.get(dnote.getId(), subject));
                }
                catch (IOException e) {
                    LOG.error("Cannot access previously listed note {} from storage ", (Object)dnote.getId(), (Object)e);
                    continue;
                }
                if (sdate.compareTo(ddate) == 0) continue;
                if (sdate.after(ddate) || this.oneWaySync) {
                    pushIDs.add(snote.getId());
                    LOG.info("Modified note is added to push list : " + sdate);
                    continue;
                }
                LOG.info("Modified note is added to pull list : " + ddate);
                pullIDs.add(snote.getId());
                continue;
            }
            pushIDs.add(snote.getId());
        }
        for (NoteInfo note : destNotes) {
            dnote = this.containsID(sourceNotes, note.getId());
            if (dnote != null) continue;
            if (this.oneWaySync) {
                LOG.info("Extraneous note is added to delete dest list : " + note.getId());
                delDstIDs.add(note.getId());
                continue;
            }
            LOG.info("Missing note is added to pull list : " + note.getId());
            pullIDs.add(note.getId());
        }
        HashMap<String, List<String>> map = new HashMap<String, List<String>>();
        map.put(pushKey, pushIDs);
        map.put(pullKey, pullIDs);
        map.put(delDstKey, delDstIDs);
        return map;
    }

    private NoteInfo containsID(List<NoteInfo> notes, String id) {
        for (NoteInfo note : notes) {
            if (!note.getId().equals(id)) continue;
            return note;
        }
        return null;
    }

    private Date lastModificationDate(Note note) {
        Date latest = new Date(0L);
        for (Paragraph paragraph : note.getParagraphs()) {
            Date tempCreated = paragraph.getDateCreated();
            Date tempStarted = paragraph.getDateStarted();
            Date tempFinished = paragraph.getDateFinished();
            if (tempCreated != null && tempCreated.after(latest)) {
                latest = tempCreated;
            }
            if (tempStarted != null && tempStarted.after(latest)) {
                latest = tempStarted;
            }
            if (tempFinished == null || !tempFinished.after(latest)) continue;
            latest = tempFinished;
        }
        return latest;
    }

    @Override
    public void close() {
        LOG.info("Closing all notebook storages");
        for (NotebookRepo repo : this.repos) {
            repo.close();
        }
    }

    public Boolean isRevisionSupportedInDefaultRepo() {
        return this.isRevisionSupportedInRepo(0);
    }

    public Boolean isRevisionSupportedInRepo(int repoIndex) {
        try {
            if (this.getRepo(repoIndex) instanceof NotebookRepoWithVersionControl) {
                return true;
            }
        }
        catch (IOException e) {
            LOG.error("Error getting default repo", (Throwable)e);
        }
        return false;
    }

    @Override
    public NotebookRepoWithVersionControl.Revision checkpoint(String noteId, String checkpointMsg, AuthenticationInfo subject) throws IOException {
        int repoCount = this.getRepoCount();
        int repoBound = Math.min(repoCount, this.getMaxRepoNum());
        int errorCount = 0;
        String errorMessage = "";
        ArrayList<NotebookRepoWithVersionControl.Revision> allRepoCheckpoints = new ArrayList<NotebookRepoWithVersionControl.Revision>();
        NotebookRepoWithVersionControl.Revision rev = null;
        for (int i = 0; i < repoBound; ++i) {
            try {
                if (!this.isRevisionSupportedInRepo(i).booleanValue()) continue;
                allRepoCheckpoints.add(((NotebookRepoWithVersionControl)this.getRepo(i)).checkpoint(noteId, checkpointMsg, subject));
                continue;
            }
            catch (IOException e) {
                LOG.warn("Couldn't checkpoint in {} storage with index {} for note {}", new Object[]{this.getRepo(i).getClass().toString(), i, noteId});
                errorMessage = errorMessage + "Error on storage class " + this.getRepo(i).getClass().toString() + " with index " + i + " : " + e.getMessage() + "\n";
                ++errorCount;
            }
        }
        if (errorCount == repoBound) {
            throw new IOException(errorMessage);
        }
        if (allRepoCheckpoints.size() > 0) {
            rev = (NotebookRepoWithVersionControl.Revision)allRepoCheckpoints.get(0);
            if (allRepoCheckpoints.size() > 1 && rev == null) {
                rev = (NotebookRepoWithVersionControl.Revision)allRepoCheckpoints.get(1);
            }
        }
        return rev;
    }

    @Override
    public Note get(String noteId, String revId, AuthenticationInfo subject) {
        Note revisionNote = null;
        try {
            if (this.isRevisionSupportedInDefaultRepo().booleanValue()) {
                revisionNote = ((NotebookRepoWithVersionControl)this.getRepo(0)).get(noteId, revId, subject);
            }
        }
        catch (IOException e) {
            LOG.error("Failed to get revision {} of note {}", new Object[]{revId, noteId, e});
        }
        return revisionNote;
    }

    @Override
    public List<NotebookRepoWithVersionControl.Revision> revisionHistory(String noteId, AuthenticationInfo subject) {
        List<NotebookRepoWithVersionControl.Revision> revisions = Collections.emptyList();
        try {
            if (this.isRevisionSupportedInDefaultRepo().booleanValue()) {
                revisions = ((NotebookRepoWithVersionControl)this.getRepo(0)).revisionHistory(noteId, subject);
            }
        }
        catch (IOException e) {
            LOG.error("Failed to list revision history", (Throwable)e);
        }
        return revisions;
    }

    @Override
    public List<NotebookRepoSettingsInfo> getSettings(AuthenticationInfo subject) {
        List<NotebookRepoSettingsInfo> repoSettings = Collections.emptyList();
        try {
            repoSettings = this.getRepo(0).getSettings(subject);
        }
        catch (IOException e) {
            LOG.error("Cannot get notebook repo settings", (Throwable)e);
        }
        return repoSettings;
    }

    @Override
    public void updateSettings(Map<String, String> settings, AuthenticationInfo subject) {
        try {
            this.getRepo(0).updateSettings(settings, subject);
        }
        catch (IOException e) {
            LOG.error("Cannot update notebook repo settings", (Throwable)e);
        }
    }

    @Override
    public Note setNoteRevision(String noteId, String revId, AuthenticationInfo subject) throws IOException {
        int repoCount = this.getRepoCount();
        int repoBound = Math.min(repoCount, this.getMaxRepoNum());
        Note currentNote = null;
        Note revisionNote = null;
        for (int i = 0; i < repoBound; ++i) {
            try {
                if (this.isRevisionSupportedInRepo(i).booleanValue()) {
                    currentNote = ((NotebookRepoWithVersionControl)this.getRepo(i)).setNoteRevision(noteId, revId, subject);
                }
            }
            catch (IOException e) {
                currentNote = null;
            }
            if (currentNote == null || revisionNote != null) continue;
            revisionNote = currentNote;
        }
        return revisionNote;
    }
}

