/*
 * Decompiled with CFR 0.152.
 */
package com.gigaspaces.internal.server.space.demote;

import com.gigaspaces.admin.demote.DemoteFailedException;
import com.gigaspaces.api.InternalApi;
import com.gigaspaces.cluster.activeelection.ISpaceModeListener;
import com.gigaspaces.cluster.activeelection.SpaceMode;
import com.gigaspaces.internal.server.space.SpaceImpl;
import com.gigaspaces.internal.utils.StringUtils;
import com.j_spaces.core.filters.ReplicationStatistics;
import java.rmi.RemoteException;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

@InternalApi
public class DemoteHandler
implements ISpaceModeListener {
    private final Logger _logger;
    private final SpaceImpl _spaceImpl;
    private final AtomicBoolean _isDemoteInProgress = new AtomicBoolean(false);
    private volatile CountDownLatch _latch;
    private final long _demoteMinTimeoutMillis;
    private static final String MIN_TIME_TO_DEMOTE_IN_MS = "engine.demote.min-timeout";
    private final long _demoteCompletionEventTimeoutMillis;

    public DemoteHandler(SpaceImpl spaceImpl) {
        this._spaceImpl = spaceImpl;
        this._logger = Logger.getLogger("com.gigaspaces.demote." + spaceImpl.getNodeName());
        this._demoteMinTimeoutMillis = StringUtils.parseDurationAsMillis(this._spaceImpl.getConfigReader().getSpaceProperty(MIN_TIME_TO_DEMOTE_IN_MS, "5s"));
        this._demoteCompletionEventTimeoutMillis = StringUtils.parseDurationAsMillis(this._spaceImpl.getConfigReader().getSpaceProperty("engine.demote.completion-event-timeout", "5s"));
    }

    public void demote(long maxSuspendTime, TimeUnit unit) throws DemoteFailedException {
        if (unit.toMillis(maxSuspendTime) < this._demoteMinTimeoutMillis) {
            throw new DemoteFailedException("Max suspend time must be equal or greater than engine.demote.min-timeout=" + this._demoteMinTimeoutMillis + "ms");
        }
        if (!this._isDemoteInProgress.compareAndSet(false, true)) {
            throw new DemoteFailedException("Demote is already in progress");
        }
        try {
            this.validationChecks();
            this._spaceImpl.addInternalSpaceModeListener(this);
            this._latch = new CountDownLatch(1);
            this.demoteImpl(maxSuspendTime, unit);
        }
        catch (TimeoutException e) {
            throw new DemoteFailedException(e.getMessage());
        }
        finally {
            this._spaceImpl.removeInternalSpaceModeListener(this);
            this._isDemoteInProgress.set(false);
        }
    }

    private long tryWithinTimeout(String msg, long timeoutMs, ParametrizedConditionProvider predicate) throws TimeoutException {
        long start = System.currentTimeMillis();
        if (!predicate.test(timeoutMs)) {
            throw new TimeoutException(msg);
        }
        long duration = System.currentTimeMillis() - start;
        long remainingTime = timeoutMs - duration;
        if (remainingTime < 0L) {
            throw new TimeoutException(msg);
        }
        return remainingTime;
    }

    private long repetitiveTryWithinTimeout(String msg, long timeoutMs, ConditionProvider f) throws TimeoutException, DemoteFailedException {
        long currTime;
        long deadline = System.currentTimeMillis() + timeoutMs;
        while ((currTime = System.currentTimeMillis()) < deadline) {
            if (f.test()) {
                return deadline - currTime;
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new DemoteFailedException("Demote got interrupted");
            }
        }
        throw new TimeoutException(msg);
    }

    private void demoteImpl(long timeout, TimeUnit timeoutUnit) throws DemoteFailedException, TimeoutException {
        long start = System.currentTimeMillis();
        long timeoutMs = timeoutUnit.toMillis(timeout);
        try {
            this._logger.info("Demoting to backup, setting suspend type to DEMOTING...");
            this._spaceImpl.getQuiesceHandler().quiesceDemote("Space is demoting from primary to backup");
            long remainingTime = timeoutMs;
            remainingTime = this.tryWithinTimeout("Couldn't demote to backup - lease manager cycle timeout", remainingTime, new ParametrizedConditionProvider(){

                @Override
                public boolean test(long innerTimeout) {
                    return DemoteHandler.this._spaceImpl.getEngine().getLeaseManager().waitForNoCycleOnQuiesce(innerTimeout);
                }
            });
            remainingTime = this.tryWithinTimeout("Couldn't demote to backup - timeout while waiting for transactions", remainingTime, new ParametrizedConditionProvider(){

                @Override
                public boolean test(long innerTimeout) {
                    return DemoteHandler.this.waitForActiveTransactions(innerTimeout);
                }
            });
            long currentDuration = System.currentTimeMillis() - start;
            if (currentDuration < this._demoteMinTimeoutMillis) {
                long timeToSleep = this._demoteMinTimeoutMillis - currentDuration;
                this._logger.info("Sleeping for [" + timeToSleep + "ms] to satisfy " + MIN_TIME_TO_DEMOTE_IN_MS);
                try {
                    Thread.sleep(timeToSleep);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new DemoteFailedException("Demote got interrupted");
                }
            }
            this.repetitiveTryWithinTimeout("Backup is not synced", remainingTime, new ConditionProvider(){

                @Override
                public boolean test() {
                    return DemoteHandler.this.isBackupSynced();
                }
            });
            this.validateSpaceStatus(false);
            this.closeOutgoingChannels();
            if (!this._spaceImpl.restartLeaderSelectorHandler()) {
                throw new DemoteFailedException("Could not restart leader selector");
            }
            try {
                boolean succeeded = this._latch.await(this._demoteCompletionEventTimeoutMillis, TimeUnit.MILLISECONDS);
                if (!succeeded) {
                    throw new DemoteFailedException("Space mode wasn't changed to be backup");
                }
                if (this._spaceImpl.getSpaceMode().equals((Object)SpaceMode.PRIMARY)) {
                    throw new DemoteFailedException("Space mode wasn't changed to backup - space still primary");
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new DemoteFailedException("Demote got interrupted");
            }
        }
        catch (DemoteFailedException e) {
            this.abort();
            throw e;
        }
        finally {
            this._logger.info("Demoting to backup finished");
            this._spaceImpl.getQuiesceHandler().unquiesceDemote();
        }
    }

    private void validateStaticConfigurations() throws DemoteFailedException {
        if (!this._spaceImpl.useZooKeeper()) {
            throw new DemoteFailedException("Primary demotion is only supported with Zookeeper leader selector.");
        }
        if (this._spaceImpl.getClusterInfo().getNumberOfBackups() != 1) {
            throw new DemoteFailedException("Cluster should be configured with exactly one backup, backups: (" + this._spaceImpl.getClusterInfo().getNumberOfBackups() + ")");
        }
        if (this._spaceImpl.getLeaderSelector() == null || !this._spaceImpl.getLeaderSelector().getClass().getName().equals("org.openspaces.zookeeper.leader_selector.ZooKeeperBasedLeaderSelectorHandler")) {
            throw new DemoteFailedException("Primary demotion is only supported with Zookeeper leader selector.");
        }
    }

    private void validateSpaceStatus(boolean checkQuiesce) throws DemoteFailedException {
        List<ReplicationStatistics.OutgoingChannel> backupChannels;
        if (!this._spaceImpl.isPrimary()) {
            throw new DemoteFailedException("Space is not primary");
        }
        if (checkQuiesce) {
            if (this._spaceImpl.getQuiesceHandler().isSuspended()) {
                throw new DemoteFailedException("Space is disconnected from ZooKeeper");
            }
            if (this._spaceImpl.getQuiesceHandler().isQuiesced()) {
                throw new DemoteFailedException("Space is quiesced");
            }
        }
        if ((backupChannels = this.getOutgoingReplication().getChannels(ReplicationStatistics.ReplicationMode.BACKUP_SPACE)).size() != 1) {
            throw new DemoteFailedException("There should be exactly one backup, current channels: (" + backupChannels.size() + ")");
        }
        ReplicationStatistics.OutgoingChannel backupChannel = backupChannels.get(0);
        if (!backupChannel.getChannelState().equals((Object)ReplicationStatistics.ChannelState.ACTIVE)) {
            throw new DemoteFailedException("Backup replication channel is not active (" + (Object)((Object)backupChannel.getChannelState()) + ")");
        }
    }

    private void validationChecks() throws DemoteFailedException {
        this.validateStaticConfigurations();
        this.validateSpaceStatus(true);
    }

    private ReplicationStatistics.OutgoingReplication getOutgoingReplication() {
        return this._spaceImpl.getEngine().getReplicationNode().getAdmin().getStatistics().getOutgoingReplication();
    }

    private void closeOutgoingChannels() {
        this._spaceImpl.getEngine().getReplicationNode().getAdmin().setPassive(false);
    }

    private void abort() {
        this._spaceImpl.getEngine().getReplicationNode().getAdmin().setActive();
    }

    @Override
    public void beforeSpaceModeChange(SpaceMode newMode) throws RemoteException {
    }

    @Override
    public void afterSpaceModeChange(SpaceMode newMode) throws RemoteException {
        if (newMode.equals((Object)SpaceMode.BACKUP)) {
            if (this._logger.isLoggable(Level.FINE)) {
                this._logger.fine("afterSpaceModeChange >> Space mode changed to backup!");
            }
        } else if (this._logger.isLoggable(Level.FINE)) {
            this._logger.fine("afterSpaceModeChange >> Unexpected Space mode changed to " + (Object)((Object)newMode));
        }
        this._latch.countDown();
    }

    private boolean isBackupSynced() {
        ReplicationStatistics.OutgoingReplication outGoingReplication = this.getOutgoingReplication();
        long lastKeyInRedoLog = outGoingReplication.getLastKeyInRedoLog();
        ReplicationStatistics.OutgoingChannel backupChannel = outGoingReplication.getChannels(ReplicationStatistics.ReplicationMode.BACKUP_SPACE).get(0);
        return backupChannel.getLastConfirmedKeyFromTarget() == lastKeyInRedoLog;
    }

    public boolean waitForActiveTransactions(long timeoutInMillis) {
        try {
            return this._spaceImpl.getEngine().getTransactionHandler().waitForActiveTransactions(timeoutInMillis);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return false;
        }
    }

    private static interface ParametrizedConditionProvider {
        public boolean test(long var1);
    }

    private static interface ConditionProvider {
        public boolean test();
    }
}

