package com.j_spaces.examples.benchmark;

import com.gigaspaces.internal.version.PlatformVersion;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFRichTextString;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.HorizontalAlignment;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.DateFormat;
import java.util.Date;

public class BenchmarkCommandReport {

    private final DateFormat dateFormat =
            DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM);

    private static final String[] resultFileColumnNames =
            {
                    "Date",//0
                    "OS Name",//1
                    "VM",//2
                    "Product Version",//3
                    "Revision",//4
                    "Operation", //5
                    "Object Type", //6
                    "Threads Number",//7
                    "Iterations Number", //8
                    "Entry Size", //9
                    "Sampling Rate", //10
                    "Batches", //11
                    "Write/Read Timeout",//12
                    "Return Lease",///13
                    "Lease Time", //14
                    "Clean Space", //15
                    "Parallel Operation",//16
                    "Number of repeats", //17
                    "Transaction", //18
                    "Commit every operation",//19
                    "Use Local Cache", //20
                    "Use SIngle Proxy", //21

                    "First Operation Average Total Time (ms)", //22
                    "First Operation Average Throughput (msg/sec)", //23
                    "First Operation Total Throughput (msg/sec)", //24
                    "Second Operation Average Total Time (m)", //25
                    "Second Operation Average Throughput (msg/sec)", //26
                    "Second Operation Total Throughput (msg/sec)", //27
                    "Third Operation Average Total Time (ms)", //28
                    "Third Operation Average Throughput (msg/sec)", //29
                    "Third Operation Total Throughput (msg/sec)" //30
            };
    private final BenchmarkCommand command;

    public BenchmarkCommandReport(BenchmarkCommand command) {
        this.command = command;
    }

    public void outDebugInfo(String resultFileName) throws FileNotFoundException, IOException {
        File file = new File(resultFileName);
        HSSFWorkbook hssfworkbook;
        // create a new sheet
        HSSFSheet sheet;
        FileOutputStream out;

        if (file.exists()) {
            POIFSFileSystem fs =
                    new POIFSFileSystem(new FileInputStream(resultFileName));

            hssfworkbook = new HSSFWorkbook(fs);
            sheet = hssfworkbook.getSheetAt(0);
            int lastRowNum = sheet.getLastRowNum();

            fillRowWithData(hssfworkbook, sheet, lastRowNum + 1);

            out = new FileOutputStream(resultFileName);
        } else {

            out = new FileOutputStream(resultFileName);
            // create a new workbook
            hssfworkbook = new HSSFWorkbook();
            // create a new sheet
            sheet = hssfworkbook.createSheet();

            // declare a row object reference
            HSSFRow r = sheet.createRow(0);

            HSSFCellStyle cs = hssfworkbook.createCellStyle();


            HSSFFont f = hssfworkbook.createFont();

            //set font 1 to 12 point type
            f.setFontHeightInPoints((short) 12);
            //make it blue
            f.setColor((short) 0xc);
            // make it bold
            //arial is the default font
            f.setBold(true);

            //set cell font
            cs.setFont(f);

            //short headerBorderType = HSSFCellStyle.BORDER_MEDIUM;
            BorderStyle borderStyle = BorderStyle.MEDIUM;
            setBorderStyle(cs, borderStyle);

            HSSFCellStyle cellStyle = hssfworkbook.createCellStyle();
            cellStyle.setAlignment(HorizontalAlignment.FILL);

            HSSFCell c;
            //create table header
            for (short cellnum = 0; cellnum < resultFileColumnNames.length; cellnum++) {
                c = r.createCell(cellnum);

                c.setCellStyle(cs);

                c.setCellValue(new HSSFRichTextString(resultFileColumnNames[cellnum]));

                sheet.setColumnWidth(cellnum, (short) 4500);
            }

            fillRowWithData(hssfworkbook, sheet, 1);
        }

        // write the workbook to the output stream
        // close our file (don't blow out our file handles
        hssfworkbook.write(out);
        out.close();
    }

    private void fillRowWithData(HSSFWorkbook wb, HSSFSheet s, int rowNumber) {
        final BenchmarkConfiguration parameters = command.getParameters();

        // declare a row object reference
        HSSFRow r = s.createRow(rowNumber);

        HSSFCellStyle cs = wb.createCellStyle();
        HSSFFont f2 = wb.createFont();

        if (rowNumber % 2 == 0) {
            cs.setFillForegroundColor(HSSFColor.GREY_25_PERCENT.index);
            cs.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        }

        BorderStyle headerBorderStyle = BorderStyle.THIN;
        setBorderStyle(cs, headerBorderStyle);

        //set font 1 to 12 point type
        f2.setFontHeightInPoints((short) 10);

        // make it bold
        //arial is the default font
        f2.setBold(true);

        //set cell stlye
        cs.setFont(f2);

        HSSFCell c;
        for (short cellnum = 0; cellnum < resultFileColumnNames.length; cellnum++) {
            c = r.createCell(cellnum);

            c.setCellStyle(cs);

            Object value;
            switch (cellnum) {
                //date
                case 0:
                    value = dateFormat.format(new Date());
                    break;

                //OS Name
                case 1:
                    value = System.getProperty("os.name");
                    break;

                //VM
                case 2:
                    value = System.getProperty("java.version");
                    break;

                //Version
                case 3:
                    value = PlatformVersion.getVersion();
                    break;

                //Build Number
                case 4:
                    value = PlatformVersion.getRevision();
                    break;

                //Operation Name
                case 5:
                    String operationName = "";
                    if (command.isFirstOperationInitialized())
                        operationName += (operationName.length() == 0 ? "" : " ") + command.getFirstOperation();
                    if (command.isSecondOperationInitialized())
                        operationName += (operationName.length() == 0 ? "" : " ") + command.getSecondOperation();
                    if (command.isThirdOperationInitialized())
                        operationName += (operationName.length() == 0 ? "" : " ") + command.getThirdOperation();

                    value = operationName;

                    break;


                //Object Type
                case 6:
                    if (command.isMap())
                        value = MessageAdaptors.getByCode(MessageAdaptors.CODE_POJO).getLabel();
                    else
                        value = MessageAdaptors.getMessageName(parameters.messageType);

                    break;

                //Threads Number
                case 7:
                    value = new Integer(parameters.threadCount);
                    break;

                //Iterations Number
                case 8:
                    value = new Integer(parameters.numberOfIterations);
                    break;

                //Entry Size
                case 9:
                    value = new Integer(parameters.messageLength);
                    break;

                //Sampling Rate
                case 10:
                    value = new Integer(parameters.globalThroughputIterations);
                    break;

                //Batches
                case 11:
                    value = new Integer(parameters.numOfBatches);
                    break;

                //Write/Read timeout
                case 12:
                    value = new Long(parameters.timeout);
                    break;


                //Return Lease
                case 13:
                    value = new Boolean(parameters.isReturnLease);
                    break;


                //Lease time value
                case 14:
                    value = new Long(parameters.lease);
                    break;

                //Clean Space
                case 15:
                    value = new Boolean(parameters.isCleanSpace);
                    break;


                //is parallel
                case 16:
                    value = new Boolean(parameters.isParallel);
                    break;

                //Read repeat numbers
                case 17:
                    value = new Long(parameters.readRepeatTimes);
                    break;

                //Transaction
                case 18:
                    if (!parameters.isSingleTransaction && parameters.spaceTransactionManager != null) {
                        if (parameters.transactionType == BenchmarkConfiguration.TRANSACTION_TYPE_DISTRIBUTED) {
                            value = "Distributed";
                        } else if (parameters.transactionType == BenchmarkConfiguration.TRANSACTION_TYPE_LOCAL) {
                            value = "Local";
                        } else {
                            value = "None";
                        }
                    } else {
                        value = "None";
                    }
                    break;

                //Commit Transaction every
                case 19:
                    if (!parameters.isSingleTransaction) {
                        value = new Long(parameters.txIterations);
                    } else {
                        value = "-";
                    }
                    break;

                //Use Local Cache
                case 20:
                    value = new Boolean(command.isUseDCache());
                    break;


                //Is Used CLustered Proxy
                case 21:
                    value = new Boolean(!parameters.isUseSingleSpaceProxy);
                    break;


                ////////////////////////////////////////////////////////////////////////
                //////************************THROUGHPUT********************************
                ////////////////////////////////////////////////////////////////////////


                //First Operation Average Total Time (ms)
                case 22:
                    value = (command.getFirstOperationAverageTotalTime() > 0) ?
                            new Double(command.getFirstOperationAverageTotalTime()) : null;
                    break;

                //First Average Throughput (msg/sec)
                case 23:
                    value = (command.getFirstAverageThroughput() > 0) ?
                            new Double(command.getFirstAverageThroughput()) : null;
                    break;

                //First Total Throughput (msg/sec)
                case 24:
                    value = (command.getFirstTotalThroughput() > 0) ?
                            new Double(command.getFirstTotalThroughput()) : null;
                    break;


                //Second Operation Average Total Time (ms)
                case 25:
                    value = (command.getSecondOperationAverageTotalTime() > 0) ?
                            new Double(command.getSecondOperationAverageTotalTime()) : null;
                    break;

                //Second Average Throughput (msg/sec)
                case 26:
                    value = (command.getSecondAverageThroughput() > 0) ?
                            new Double(command.getSecondAverageThroughput()) : null;
                    break;

                //Second Total Throughput (msg/sec)
                case 27:
                    value = (command.getSecondTotalThroughput() > 0) ?
                            new Double(command.getSecondTotalThroughput()) : null;
                    break;


                //Third Operation Average Total Time (ms)
                case 28:
                    value = (command.getThirdOperationAverageTotalTime() > 0) ?
                            new Double(command.getThirdOperationAverageTotalTime()) : null;

                    break;

                //Third Average Throughput (msg/sec)
                case 29:
                    value = (command.getThirdAverageThroughput() > 0) ?
                            new Double(command.getThirdAverageThroughput()) : null;

                    break;

                //Third Total Throughput (msg/sec)
                case 30:
                    value = (command.getThirdTotalThroughput() > 0) ?
                            new Double(command.getThirdTotalThroughput()) : null;

                    break;

                default:
                    value = "-";
            }

            if (value instanceof String) {
                c.setCellValue(new HSSFRichTextString((String) value));
            } else if (value instanceof Number) {
                c.setCellValue(((Number) value).doubleValue());
            } else if (value instanceof Date) {
                c.setCellValue((Date) value);
            } else if (value instanceof Boolean) {
                c.setCellValue(((Boolean) value).booleanValue());
            }
        }
    }

    private void setBorderStyle(HSSFCellStyle cs, BorderStyle borderStyle) {
        cs.setBorderBottom(borderStyle);
        cs.setBorderLeft(borderStyle);
        cs.setBorderRight(borderStyle);
        cs.setBorderTop(borderStyle);
    }
}
