/*
 * @(#)BenchmarkTestResult.java   Sep 10, 2007
 *
 * Copyright 2007 GigaSpaces Technologies Inc.
 */

package com.j_spaces.examples.benchmark;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * BenchmarkTestResult is a result object containing the result of a benchmark run. It holds a list
 * of {@link Result} objects, each representing a specific operation run. A Result may hold
 * benchmark data returned by each thread. The result object holds a list of {@link ThreadResult}
 * objects, each representing a result of a thread operation run.
 *
 * @since 6.0.2
 */
public class BenchmarkTestResult {
    /**
     * Models a benchmark operation result per thread, each containing throughput or the thread, and
     * duration executing the operation by the thread.
     */
    private static final class ThreadResult {
        final double threadTP;
        final long threadDuration;

        /**
         * Constructs a result for a thread.
         *
         * @param threadTP       The thread's throughput in msg/sec.
         * @param threadDuration The thread'd duration in milliseconds.
         */
        public ThreadResult(double threadTP, long threadDuration) {
            this.threadTP = threadTP;
            this.threadDuration = threadDuration;
        }

        /*
         * @see java.lang.Object#toString()
         */
        @Override
        public String toString() {
            return "TP: " + threadTP + " msg/sec" + " Duration: " + threadDuration + " ms";
        }
    }

    /**
     * Models a benchmark operation result, each containing throughput of either 1 or more threads,
     * average duration time of all threads, average throughput of all threads, total throughput of
     * all threads.
     */
    public static final class Result {
        /**
         * representation of this benchmark result
         */
        private String key;

        /**
         * throughput of each thread (1 or more)
         */
        private List<ThreadResult> threadResult = new LinkedList<ThreadResult>();

        /**
         * average duration time of all threads
         */
        private double avgDuration;

        /**
         * average throughput of all threads
         */
        private double avgTP;

        /**
         * total throughput of all threads
         */
        private double totalTP;

        /**
         * Construct a Result object with a unique key.
         *
         * @param key Unique key.
         */
        public Result(String key) {
            this.key = key;
        }

        /**
         * Adds a result for a thread.
         *
         * @param threadTP Throughput of thread.
         * @param duration Duration time of thread.
         */
        void addThreadResult(double threadTP, long duration) {
            threadResult.add(new ThreadResult(threadTP, duration));
        }

        /**
         * Set the average duration time of all threads.
         *
         * @param duration Duration in ms.
         */
        void setAverageDuration(double duration) {
            this.avgDuration = duration;
        }

        /**
         * Sets the average throughput of all threads.
         *
         * @param throughput Throughput msg/sec.
         */
        void setAverageThroughput(double throughput) {
            this.avgTP = throughput;
        }

        /**
         * Sets the total throughput of all threads.
         *
         * @param total Total throughput in msg/sec.
         */
        void setTotalThroughput(double total) {
            this.totalTP = total;
        }

        /**
         * @return The unique description of this result.
         */
        public String getKey() {
            return this.key;
        }

        /**
         * @return The average duration [ms].
         */
        public double getAvgDuration() {
            return this.avgDuration;
        }

        /**
         * @return The average throughput [msg/sec].
         */
        public double getAvgThroughput() {
            return this.avgTP;
        }

        /**
         * @return The total throughput of all threads [msg/sec].
         */
        public double getTotalThroughput() {
            return this.totalTP;
        }

        /*
         * @see java.lang.Object#toString()
         */
        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();

            sb.append("Operation: " + key + "\n");

            for (int i = 0; i < threadResult.size(); ++i)
                sb.append("Thread [" + i + "] " + threadResult.get(i) + "\n");

            sb.append("Average duration for all threads: " + avgDuration + " ms\n");
            sb.append("Average throughput for all threads: " + avgTP + " msg/sec\n");
            sb.append("Total throughput for all threads: " + totalTP + " msg/sec\n");

            return sb.toString();
        }
    }

    /**
     * input arguments used to generate this result
     */
    private Map<String, String> inputArgs;

    /**
     * results of all operations conducted by this benchmark test
     */
    private List<Result> results = new LinkedList<Result>();

    /**
     * Exception if benchmark failed
     */
    private Exception exception;

    /**
     * @param args Input arguments used to execute this benchmark.
     */
    public void setInputArguments(Map<String, String> args) {
        this.inputArgs = args;
    }

    /**
     * @return The Exception caught while running the benchmark.
     */
    public Exception getException() {
        return exception;
    }

    /**
     * Set Exception if benchmark failed.
     *
     * @param exception Exception to set.
     */
    public void setException(Exception exception) {
        this.exception = exception;
    }

    /**
     * Builds a result object by description key.
     *
     * @param description description of this result.
     * @return Result object.
     */
    public Result buildResult(String description) {
        Result newResult = new Result(description);
        results.add(newResult);
        return newResult;
    }

    /**
     * Returns the average throughput between all benchmark test results.
     */
    public Map<String, Result> aggregate() {
        //TODO - remove hack code when full warmup implementation is implemented.
        final int warmups = 2; //declare a warmups count (HACK #1)

        Map<String, Result> aggregatedMap = new HashMap<String, Result>();
        Map<String, Integer> totalResults = new HashMap<String, Integer>();

        /*
         * for each result in the results list, aggregate it by key
         */
        for (Result result : results) {
            Result aggregatedResult = aggregatedMap.get(result.key);
            if (aggregatedResult == null) {
                aggregatedResult = new Result(result.key);
                aggregatedMap.put(result.key, aggregatedResult);
                totalResults.put(result.key, 0);
            }

            //only include results from a repeat cycle, otherwise skip it
            if (totalResults.get(result.key) >= warmups) //is this result from a repeat cycle? (HACK #2)
            {
                aggregatedResult.avgDuration += result.avgDuration;
                aggregatedResult.avgTP += result.avgTP;
                aggregatedResult.totalTP += result.totalTP;
            }
            totalResults.put(result.key, totalResults.get(result.key) + 1);
        }

        /*
         * Compute average
         */
        for (Result aggregatedResult : aggregatedMap.values()) {
            Integer total = new Integer(totalResults.get(aggregatedResult.key).intValue() - warmups); //don't consider warmups in total (HACK #3)
            aggregatedResult.avgDuration /= total;
            aggregatedResult.avgTP /= total;
            aggregatedResult.totalTP /= total;
        }

        return aggregatedMap;
    }

    /**
     * Aggregates as in {@link #aggregate()} but returns the result of the last operation performed
     * by the benchmark. e.g., -write -take will result in a take, -all will result in a take also,
     * -write -read will result in a read.
     *
     * @return last aggregated result.
     */
    public Result aggregateByLast() {
        Map<String, Result> aggregatedMap = aggregate();
        Result lastResult = results.get(results.size() - 1);

        Result lastAggregatedResult = aggregatedMap.get(lastResult.key);
        return lastAggregatedResult;
    }

    /**
     * Representation of this instance. Displaying the exception if any, input arguments, and the
     * for each {@link Result} it's toString().
     *
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        if (exception != null)
            return exception.toString();

        StringBuilder sb = new StringBuilder();

        sb.append("Input arguments: ");
        sb.append(inputArgs);
        sb.append("\n\n");

        for (Result result : results) {
            sb.append(result);
            sb.append("\n");
        }

        return sb.toString();
    }
}
